% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sig_tally.R
\name{sig_tally}
\alias{sig_tally}
\alias{sig_tally.CopyNumber}
\alias{sig_tally.MAF}
\title{Tally a Genomic Alteration Object}
\usage{
sig_tally(object, ...)

\method{sig_tally}{CopyNumber}(
  object,
  method = "Wang",
  ignore_chrs = NULL,
  feature_setting = sigminer::CN.features,
  type = c("probability", "count"),
  reference_components = FALSE,
  cores = 1,
  seed = 123456,
  min_comp = 2,
  max_comp = 15,
  min_prior = 0.001,
  model_selection = "BIC",
  threshold = 0.1,
  nrep = 1,
  niter = 1000,
  keep_only_matrix = FALSE,
  ...
)

\method{sig_tally}{MAF}(
  object,
  ref_genome = NULL,
  prefix = NULL,
  add = TRUE,
  ignore_chrs = NULL,
  use_syn = TRUE,
  keep_only_matrix = FALSE,
  ...
)
}
\arguments{
\item{object}{a \link{CopyNumber} object or \link{MAF} object.}

\item{...}{custom setting for operating object. Detail see S3 method for
corresponding class (e.g. \code{CopyNumber}).}

\item{method}{method for feature classfication, can be one of "Macintyre" ("M") and
"Wang" ("W").}

\item{ignore_chrs}{Chromsomes to ignore from analysis. e.g. chrX and chrY.}

\item{feature_setting}{a \code{data.frame} used for classification.
Only used when method is "Wang" ("W").
Default is \link{CN.features}. Users can also set custom input with "feature",
"min" and "max" columns available. Valid features can be printed by
\code{unique(CN.features$feature)}.}

\item{type}{one of "probability", "count". Default is "probability", return a matrix
with the sum of posterior probabilities for each components. If set to 'count',
return a matrix with event count assigned to each components. The result for
both types should be close. Only used when method is "Macintyre".}

\item{reference_components}{default is \code{FALSE}, calculate mixture components
from \link{CopyNumber} object. Only used when method is "Macintyre".}

\item{cores}{number of compute cores to run this task.
You can use \code{\link[future:availableCores]{future::availableCores()}} function to check how
many cores you can use.}

\item{seed}{seed number. Only used when method is "Macintyre".}

\item{min_comp}{minimal number of components to fit, default is 2.
Can also be a vector with length 6, which apply to each feature.
Only used when method is "Macintyre".}

\item{max_comp}{maximal number of components to fit, default is 15.
Can also be a vector with length 6, which apply to each feature.
Only used when method is "Macintyre".}

\item{min_prior}{the minimum relative size of components, default is 0.001.
Details about custom setting please refer to \strong{flexmix} package.
Only used when method is "Macintyre".}

\item{model_selection}{model selection strategy, default is 'BIC'.
Details about custom setting please refer to \strong{flexmix} package.
Only used when method is "Macintyre".}

\item{threshold}{default is \code{0.1}. Sometimes, the result components
include adjacent distributions with similar mu
(two and more distribution are very close), we use this threshold
to obtain a more meaningful fit with less components.
Only used when method is "Macintyre".}

\item{nrep}{number of run times for each value of component,
keep only the solution with maximum likelihood.
Only used when method is "Macintyre".}

\item{niter}{the maximum number of iterations.
Only used when method is "Macintyre".}

\item{keep_only_matrix}{if \code{TRUE}, keep only matrix for signature extraction.}

\item{ref_genome}{BSgenome object or name of the installed BSgenome package. Example: BSgenome.Hsapiens.UCSC.hg19
Default NULL, tries to auto-detect from installed genomes.}

\item{prefix}{Prefix to add or remove from contig names in MAF file.}

\item{add}{If prefix is used, default is to add prefix to contig names in MAF file. If false prefix will be removed from contig names.}

\item{use_syn}{Logical. Whether to include synonymous variants in analysis. Defaults to TRUE}
}
\value{
a \code{list} contains a \code{matrix} used for NMF de-composition.
}
\description{
Tally a variation object like \link{MAF} and return a matrix for NMF de-composition and more.
This is a generic function,
so it can be further extended to other mutation cases. Please read details
about how to set sex for identifying copy number signatures.
}
\details{
For identifying copy number signatures, we have to derive copy number
features firstly. Due to the difference of copy number values in sex chromosomes
between male and female, we have to do an extra step \strong{if we don't want to
ignore them}.

I create two options to control this, the default values are shown as
the following, you can use the same way to set (per R session).

\code{options(sigminer.sex = "female", sigminer.copynumber.max = NA_integer_)}
\itemize{
\item If your cohort are all females, you can totally ignore this.
\item If your cohort are all males, set \code{sigminer.sex} to 'male' and
\code{sigminer.copynumber.max} to a proper value (the best is consistent
with \link{read_copynumber}).
\item If your cohort contains bother males and females, set \code{sigminer.sex}
as a \code{data.frame} with two columns "sample" and "sex". And
set \code{sigminer.copynumber.max} to a proper value (the best is consistent
with \link{read_copynumber}).
}
}
\section{Methods (by class)}{
\itemize{
\item \code{CopyNumber}: Returns copy number features, components and component-by-sample matrix

\item \code{MAF}: Returns SBS mutation component-by-sample matrix and APOBEC enrichment
}}

\examples{
# Load copy number object
load(system.file("extdata", "toy_copynumber.RData",
  package = "sigminer", mustWork = TRUE
))
\donttest{
# Use method designed by Wang, Shixiang et al.
cn_tally_W <- sig_tally(cn, method = "W")
# Use method designed by Macintyre et al.
cn_tally_M <- sig_tally(cn, method = "M")
}
\donttest{
# Prepare SBS signature analysis
laml.maf <- system.file("extdata", "tcga_laml.maf.gz", package = "maftools")
laml <- read_maf(maf = laml.maf)
if (require("BSgenome.Hsapiens.UCSC.hg19")) {
  mt_tally <- sig_tally(
    laml,
    ref_genome = "BSgenome.Hsapiens.UCSC.hg19",
    prefix = "chr", add = TRUE, useSyn = TRUE
  )
} else {
  message("Please install package 'BSgenome.Hsapiens.UCSC.hg19' firstly!")
}
}
}
\references{
Macintyre, Geoff, et al. "Copy number signatures and mutational
processes in ovarian carcinoma." Nature genetics 50.9 (2018): 1262.

Mayakonda, Anand, et al. "Maftools: efficient and comprehensive analysis of somatic variants in cancer." Genome research 28.11 (2018): 1747-1756.
}
\seealso{
\link{sig_estimate} for estimating signature number for \link{sig_extract},
\link{sig_auto_extract} for extracting signatures using automatic relevance determination technique.
}
\author{
Shixiang Wang
}
