% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/intra_clade_phyglm.R
\name{intra_clade_phyglm}
\alias{intra_clade_phyglm}
\title{Interaction between intraspecific variability and influential clade detection - Phylogenetic Logistic Regression}
\usage{
intra_clade_phyglm(formula, data, phy, clade.col, n.species = 5,
  n.sim = 100, n.intra = 2, Vx = NULL, distrib = "normal",
  x.transf = NULL, btol = 50, track = TRUE, ...)
}
\arguments{
\item{formula}{The model formula}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{clade.col}{The column in the provided data frame which specifies the
clades (a character vector with clade names).}

\item{n.species}{Minimum number of species in a clade for the clade to be
included in the leave-one-out deletion analysis. Default is \code{5}.}

\item{n.sim}{Number of simulations for the randomization test.}

\item{n.intra}{Number of datasets resimulated taking into account intraspecific variation (see: \code{"intra_phyglm"})}

\item{Vx}{Name of the column containing the standard deviation or the standard error of the predictor 
variable. When information is not available for one taxon, the value can be 0 or \code{NA}}

\item{distrib}{A character string indicating which distribution to use to generate a random value for the response 
and/or predictor variables. Default is normal distribution: "normal" (function \code{\link{rnorm}}).
Uniform distribution: "uniform" (\code{\link{runif}})
Warning: we recommend to use normal distribution with Vx or Vy = standard deviation of the mean.}

\item{x.transf}{Transformation for the predictor variable (e.g. \code{"log"} or \code{"sqrt"}). Please use this 
argument instead of transforming data in the formula directly (see also details below).}

\item{btol}{Bound on searching space. For details see \code{phyloglm}}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{phyloglm}}
}
\value{
The function \code{intra_clade_phyglm} returns a list with the following
components:

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (e.g. \code{lambda}) for the full model
without deleted species.

\code{sensi.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade. Columns report the calculated
regression intercept (\code{intercept}), difference between simulation
intercept and full model intercept (\code{DIFintercept}), the percentage of change
in intercept compared to the full model (\code{intercept.perc}) and intercept
p-value (\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DIFestimate} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter 
(e.g. \code{kappa} or \code{lambda}, depending on the phylogenetic model used) 
are reported.

\code{null.dist}: A data frame with estimates for the null distributions
for all clades analysed.

\code{data}: Original full dataset.

\code{errors}: Clades and/or iterations where deletion resulted in errors.
}
\description{
Estimate the impact on model estimates of phylogenetic logistic regression after 
removing clades from the analysis, while taking into account potential
interactions with intraspecific variability.
}
\details{
This function sequentially removes one clade at a time, fits a phylogenetic
logistic regression model using \code{\link[phylolm]{phyloglm}} and stores the
results. The impact of of a specific clade on model estimates is calculated by the
comparison between the full model (with all species) and the model without 
the species belonging to a clade. This operation is repeated \code{n.intra} times for
simulated values of the dataset, taking into account intraspecific variation. At each iteration, the function 
generates a random value for each row in the dataset using the standard deviation or errors supplied, and 
detect the influential species within that iteration. 

Additionally, to account for the influence of the number of species on each 
clade (clade sample size), this function also estimate a null distribution 
expected for the number of species in a given clade. This is done by fitting
models without the same number of species in the given clade. 
 The number of simulations to be performed is set by 'n.sim'. To test if the 
 clade influence differs from the null expectation for a clade of that size, 
 a randomization test can be performed using 'summary(x)'. 

All phylogenetic models from \code{phyloglm} can be used, i.e. \code{BM},
\code{OUfixedRoot}, \code{OUrandomRoot}, \code{lambda}, \code{kappa},
\code{delta}, \code{EB} and \code{trend}. See ?\code{phyloglm} for details.

\code{clade_phyglm} detects influential clades based on
difference in intercept and/or slope when removing a given clade compared
to the full model including all species.

Currently, this function can only implement simple logistic models (i.e. 
\eqn{y = a + bx}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
set.seed(6987)
phy = rtree(100)
x = rTrait(n=1,phy=phy,parameters=list(ancestral.state=2,optimal.value=2,sigma2=1,alpha=1))
X = cbind(rep(1,100),x)
y = rbinTrait(n=1,phy=phy, beta=c(-1,0.5), alpha=.7 ,X=X)
z = rnorm(n = length(x),mean = mean(x),sd = 0.1*mean(x))
cla <- rep(c("A","B","C","D"), each = 25)
dat = data.frame(y, x, z, cla)
intra_clade <- intra_clade_phyglm(formula=y ~ x, data = dat, phy = phy,
                                  clade.col = "cla", n.sim = 30, n.intra = 3,
                                  x.transf = log, Vx = "z", distrib="normal")
sensi_plot(intra_clade)
sensi_plot(intra_clade, clade = "B", graphs = 2)
}
\dontshow{
set.seed(6987)
phy = rtree(100)
x = rTrait(n=1,phy=phy,parameters=list(ancestral.state=2,optimal.value=2,sigma2=1,alpha=1))
X = cbind(rep(1,100),x)
y = rbinTrait(n=1,phy=phy, beta=c(-1,0.5), alpha=.7 ,X=X)
z = rnorm(n = length(x),mean = mean(x),sd = 0.1*mean(x))
cla <- rep(c("A","B","C","D"), each = 25)
dat = data.frame(y, x, z, cla)
intra_clade <- intra_clade_phyglm(formula=y ~ x, data = dat, phy = phy,
                                 clade.col = "cla", n.sim = 1, n.intra = 1,
                                 x.transf = log, Vx = "z", distrib="normal")
sensi_plot(intra_clade)
}
}
\references{
Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phyloglm}}, \code{\link[sensiPhy]{intra_phyglm}},
\code{\link{clade_phyglm}},  \code{\link{intra_clade_phylm}}, 
\code{\link{sensi_plot}}
}
\author{
Gustavo Paterno, Caterina Penone
}
