% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/route.R
\name{Route}
\alias{Route}
\title{Single route dispatch}
\description{
Class for handling a single route dispatch
}
\details{
The \code{Route} class is used to encapsulate a single URL dispatch, that is,
chose a single handler from a range based on a URL path. A handler will be
called with a request, response, and keys argument as well as any additional
arguments passed on to \code{dispatch()}.

The path will strip the query string prior to assignment of the handler, can
contain wildcards, and can be parameterised using the \code{:} prefix. If there
are multiple matches of the request path the most specific will be chosen.
Specificity is based on number of elements (most), number of parameters
(least), and number of wildcards (least), in that order. Parameter
values will be available in the keys argument passed to the handler, e.g. a
path of \verb{/user/:user_id} will provide \code{list(user_id = 123)} for a dispatch on
\verb{/user/123} in the \code{keys} argument.

Handlers are only called for their side-effects and are expected to return
either \code{TRUE} or \code{FALSE} indicating whether additional routes in a
\code{\link{RouteStack}} should be called, e.g. if a handler is returning \code{FALSE} all
further processing of the request will be terminated and the response will be
passed along in its current state. Thus, the intend of the handlers is to
modify the request and response objects, in place. All calls to handlers will
be wrapped in \code{\link[=try]{try()}} and if an exception is raised the response code will be
set to \code{500} with the body of the response being the error message. Further
processing of the request will be terminated. If a different error handling
scheme is wanted it must be implemented within the handler (the standard
approach is chosen to avoid handler errors resulting in a server crash).

A handler is referencing a specific HTTP method (\code{get}, \code{post}, etc.) but can
also reference \code{all} to indicate that it should match all types of requests.
Handlers referencing \code{all} have lower precedence than those referencing
specific methods, so will only be called if a match is not found within the
handlers of the specific method.
}
\section{Initialization}{

A new 'Route'-object is initialized using the \code{new()} method on the
generator or alternatively by using \code{\link[=route]{route()}}:

\strong{Usage}
\tabular{l}{
\code{route <- Route$new(...)}
}
\tabular{l}{
\code{route <- route(...)}
}
}

\examples{
# Initialise an empty route
route <- Route$new()

# Initialise a route with handlers assigned
route <- Route$new(
  all = list(
    '/*' = function(request, response, keys, ...) {
      message('Request received')
      TRUE
    }
  )
)

# Remove it again
route$remove_handler('all', '/*')

}
\seealso{
\link{RouteStack} for binding multiple routes sequentially
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{root}}{The root of the route. Will be removed from the path of any
request before matching a handler}

\item{\code{name}}{An autogenerated name for the route}

\item{\code{empty}}{Is the route empty}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Route-new}{\code{Route$new()}}
\item \href{#method-Route-print}{\code{Route$print()}}
\item \href{#method-Route-add_handler}{\code{Route$add_handler()}}
\item \href{#method-Route-remove_handler}{\code{Route$remove_handler()}}
\item \href{#method-Route-get_handler}{\code{Route$get_handler()}}
\item \href{#method-Route-remap_handlers}{\code{Route$remap_handlers()}}
\item \href{#method-Route-merge_route}{\code{Route$merge_route()}}
\item \href{#method-Route-dispatch}{\code{Route$dispatch()}}
\item \href{#method-Route-on_attach}{\code{Route$on_attach()}}
\item \href{#method-Route-clone}{\code{Route$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-new"></a>}}
\if{latex}{\out{\hypertarget{method-Route-new}{}}}
\subsection{Method \code{new()}}{
Create a new Route
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$new(..., ignore_trailing_slash = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Handlers to add up front. Must be in the form of named lists
where the names corresponds to paths and the elements are the handlers.
The name of the argument itself defines the method to listen on (see
examples)}

\item{\code{ignore_trailing_slash}}{Logical. Should the trailing slash of a path
be ignored when adding handlers and handling requests. Setting this will
not change the request or the path associated with but just ensure that
both \code{path/to/resource} and \verb{path/to/resource/} ends up in the same
handler. Because the request is left untouched, setting this to \code{TRUE}
will not affect further processing by other routes}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-print"></a>}}
\if{latex}{\out{\hypertarget{method-Route-print}{}}}
\subsection{Method \code{print()}}{
Pretty printing of the object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Ignored}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-add_handler"></a>}}
\if{latex}{\out{\hypertarget{method-Route-add_handler}{}}}
\subsection{Method \code{add_handler()}}{
Add a handler to the specified method and path. The special
method \code{'all'} will allow the handler to match all http request methods.
The path is a URL path consisting of strings, parameters (strings
prefixed with \code{:}), and wildcards (\code{*}), separated by \code{/}. A wildcard
will match anything and is thus not restricted to a single path element
(i.e. it will span multiple \code{/} if possible). The handler must be a
function containing the arguments \code{request}, \code{response}, \code{keys}, and
\code{...}, and must return either \code{TRUE} or \code{FALSE}. The \code{request} argument
will be a \link[reqres:Request]{reqres::Request} object and the \code{response} argument will be a
\link[reqres:Response]{reqres::Response} object matching the current exchange. The \code{keys}
argument will be a named list with the value of all matched parameters
from the path. Any additional argument passed on to the \code{dispatch} method
will be avaiable as well. This method will override an existing handler
with the same method and path.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$add_handler(method, path, handler, reject_missing_methods = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{The http method to match the handler to}

\item{\code{path}}{The URL path to match to}

\item{\code{handler}}{A handler function}

\item{\code{reject_missing_methods}}{Should requests to this path that doesn't
have a handler for the specific method automatically be rejected with a
405 Method Not Allowed response with the correct Allow header informing
the client of the implemented methods. Assigning a handler to \code{"all"} for
the same path at a later point will overwrite this functionality. Be
aware that setting this to \code{TRUE} will prevent the request from falling
through to other routes that might have a matching method and path.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-remove_handler"></a>}}
\if{latex}{\out{\hypertarget{method-Route-remove_handler}{}}}
\subsection{Method \code{remove_handler()}}{
Removes the handler assigned to the specified method and
path. If no handler have been assigned it will silently ignore it.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$remove_handler(method, path)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{The http method of the handler to remove}

\item{\code{path}}{The URL path of the handler to remove}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-get_handler"></a>}}
\if{latex}{\out{\hypertarget{method-Route-get_handler}{}}}
\subsection{Method \code{get_handler()}}{
Returns a handler already assigned to the specified method
and path. If no handler have been assigned it will return NULL.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$get_handler(method, path)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{method}}{The http method of the handler to find}

\item{\code{path}}{The URL path of the handler to find}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-remap_handlers"></a>}}
\if{latex}{\out{\hypertarget{method-Route-remap_handlers}{}}}
\subsection{Method \code{remap_handlers()}}{
Allows you to loop through all added handlers and reassings
them at will. A function with the parameters \code{method}, \code{path}, and
\code{handler} must be provided which is responsible for reassigning the
handler given in the arguments. If the function does not reassign the
handler, then the handler is removed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$remap_handlers(.f)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{.f}}{A function performing the remapping of each handler}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-merge_route"></a>}}
\if{latex}{\out{\hypertarget{method-Route-merge_route}{}}}
\subsection{Method \code{merge_route()}}{
Merge another route into this one, adopting all its handlers.
The other route will be empty after the merge.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$merge_route(route, use_root = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{route}}{A Route object}

\item{\code{use_root}}{Should the root of \code{route} be prepended to all paths from
the route before adding them}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-dispatch"></a>}}
\if{latex}{\out{\hypertarget{method-Route-dispatch}{}}}
\subsection{Method \code{dispatch()}}{
Based on a \link[reqres:Request]{reqres::Request} object the route will find the
correct handler and call it with the correct arguments. Anything passed
in with \code{...} will be passed along to the handler.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$dispatch(request, ..., .require_bool_output = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{request}}{The request to route}

\item{\code{...}}{Additional arguments to the handlers}

\item{\code{.require_bool_output}}{Should the dispatch enforce a boolean output.
Mainly for internal use.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-on_attach"></a>}}
\if{latex}{\out{\hypertarget{method-Route-on_attach}{}}}
\subsection{Method \code{on_attach()}}{
Method for use by \code{fiery} when attached as a plugin. Should
not be called directly. This method creates a RouteStack with the route
as the single route and then mounts that to the app. For more flexibility
create the RouteStack manually
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$on_attach(app, on_error = deprecated(), ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{app}}{The Fire object to attach the router to}

\item{\code{on_error}}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} A function for error handling}

\item{\code{...}}{Ignored}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Route-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Route-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Route$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
