% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rotate.R, R/rotate_date.R, R/rotate_time.R
\name{rotate}
\alias{rotate}
\alias{backup}
\alias{prune_backups}
\alias{rotate_date}
\alias{backup_date}
\alias{rotate_time}
\alias{backup_time}
\title{Rotate or backup files}
\usage{
rotate(
  file,
  size = 1,
  max_backups = Inf,
  compression = FALSE,
  backup_dir = dirname(file),
  create_file = TRUE,
  dry_run = FALSE,
  verbose = dry_run
)

backup(
  file,
  size = 0,
  max_backups = Inf,
  compression = FALSE,
  backup_dir = dirname(file),
  dry_run = FALSE,
  verbose = dry_run
)

prune_backups(
  file,
  max_backups,
  backup_dir = dirname(file),
  dry_run = FALSE,
  verbose = dry_run
)

rotate_date(
  file,
  age = 1,
  size = 1,
  max_backups = Inf,
  compression = FALSE,
  format = "\%Y-\%m-\%d",
  backup_dir = dirname(file),
  overwrite = FALSE,
  create_file = TRUE,
  now = Sys.Date(),
  dry_run = FALSE,
  verbose = dry_run
)

backup_date(
  file,
  age = 1,
  size = 1,
  max_backups = Inf,
  compression = FALSE,
  format = "\%Y-\%m-\%d",
  backup_dir = dirname(file),
  overwrite = FALSE,
  now = Sys.Date(),
  dry_run = FALSE,
  verbose = dry_run
)

rotate_time(
  file,
  age = -1,
  size = 1,
  max_backups = Inf,
  compression = FALSE,
  format = "\%Y-\%m-\%d--\%H-\%M-\%S",
  backup_dir = dirname(file),
  overwrite = FALSE,
  create_file = TRUE,
  now = Sys.time(),
  dry_run = FALSE,
  verbose = dry_run
)

backup_time(
  file,
  age = -1,
  size = 1,
  max_backups = Inf,
  compression = FALSE,
  format = "\%Y-\%m-\%d--\%H-\%M-\%S",
  backup_dir = dirname(file),
  overwrite = FALSE,
  now = Sys.time(),
  dry_run = FALSE,
  verbose = dry_run
)
}
\arguments{
\item{file}{\code{character} scalar: file to backup/rotate}

\item{size}{scalar \code{integer}, \code{character} or \code{Inf}. Backup/rotate only if
\code{file} is larger than this size. \code{Integers} are interpreted as bytes. You
can pass \code{character} vectors that contain a file size suffix like \verb{1k}
(kilobytes), \verb{3M} (megabytes), \verb{4G} (gigabytes), \verb{5T} (terabytes). Instead
of these short forms you can also be explicit and use the IEC suffixes
\code{KiB}, \code{MiB}, \code{GiB}, \code{TiB}. In Both cases \code{1} kilobyte is \code{1024} bytes, 1
\code{megabyte} is \code{1024} kilobytes, etc... .}

\item{max_backups}{maximum number of backups to keep
\itemize{
\item an \code{integer} scalar: Maximum number of backups to keep
}

In addition for timestamped backups the following value are supported:
\itemize{
\item a \code{Date} scalar: Remove all backups before this date
\item a \code{character} scalar representing a Date in ISO format
(e.g. \code{"2019-12-31"})
\item a \code{character} scalar representing an Interval in the form
\code{"<number> <interval>"} (see below for more info)
}}

\item{compression}{Whether or not backups should be compressed
\itemize{
\item \code{FALSE} for uncompressed backups,
\item \code{TRUE} for zip compression; uses \code{\link[=zip]{zip()}}
\item a scalar \code{integer} between \code{1} and \code{9} to specify a compression
level (requires the
\href{https://CRAN.R-project.org/package=zip}{zip} package,
see its documentation for details)
\item the \code{character} scalars \code{"utils::zip()"} or \code{"zip::zipr"} to force a
specific zip command
}}

\item{backup_dir}{\code{character} scalar. The directory in which the backups
of \code{file} are stored (defaults to \code{dirname(file)})}

\item{create_file}{\code{logical} scalar. If \code{TRUE} create an empty file in
place of \code{file} after rotating.}

\item{dry_run}{\code{logical} scalar. If \code{TRUE} no changes are applied to the
file system (no files are created or deleted)}

\item{verbose}{\code{logical} scalar. If \code{TRUE} additional informative \code{messages}
are printed}

\item{age}{minimum age after which to backup/rotate a file; can be
\itemize{
\item a \code{character} scalar representing an Interval in the form
\code{"<number> <interval>"} (e.g. \code{"2 months"}, see \emph{Intervals} section below).
\item a \code{Date} or a \code{character} scalar representing a Date for
a fixed point in time after which to backup/rotate. See \code{format} for
which Date/Datetime formats are supported by rotor.
}}

\item{format}{a scalar \code{character} that can be a subset of of valid
\code{strftime()} formatting strings. The default setting is
\code{"\%Y-\%m-\%d--\%H-\%M-\%S"}.
\itemize{
\item You can use an arbitrary number of dashes anywhere in the format, so
\code{"\%Y-\%m-\%d--\%H-\%M-\%S"} and \code{"\%Y\%m\%d\%H\%M\%S"} are both legal.
\item \code{T} and \verb{_} can also be used as separators. For example, the following
datetime formats are also possible:
\verb{\%Y-\%m-\%d_\%H-\%M-\%S} (Python logging default),
\verb{\%Y\%m\%dT\%H\%M\%S} (\href{https://en.wikipedia.org/wiki/ISO_8601}{ISO 8601})
\item All datetime components except \verb{\%Y} are optional. If you leave out part
of the timestamp, the first point in time in the period is assumed. For
example (assuming the current year is 2019) \verb{\%Y} is identical to
\code{2019-01-01--00-00-00}.
\item The timestamps must be lexically sortable, so \code{"\%Y-\%m-\%d"} is legal,
\code{"\%m-\%d-\%Y"} and \verb{\%Y-\%d} are not.
}}

\item{overwrite}{\code{logical} scalar. If \code{TRUE} overwrite backups if a backup
of the same name (usually due to timestamp collision) exists.}

\item{now}{The current \code{Date} or time (\code{POSIXct}) as a scalar. You can pass a
custom value here to to override the real system time. As a convenience you
can also pass in \code{character} strings that follow the guidelines outlined
above for \code{format}, but please note that these differ from the formats
understood by \code{\link[=as.POSIXct]{as.POSIXct()}} or \code{\link[=as.Date]{as.Date()}}.}
}
\value{
\code{file} as a \code{character} scalar (invisibly)
}
\description{
Functions starting with \code{backup} create backups of a \code{file}, while functions
starting with \code{rotate} do the same but also replace the original \code{file}
with an empty one (this is useful for log rotation)

\code{prune_backups()} physically deletes all backups of a file
based on \code{max_backups}
}
\section{Side Effects}{

\code{backup()}, \code{backup_date()}, and \code{backup_time()} may create files (if the
specified conditions are met). They may also delete backups, based on
\code{max_backup}.

\code{rotate()}, \code{rotate_date()} and \code{rotate_time()} do the same, but in
addition delete the input \code{file}, or replace it with an empty file if
\code{create_file == TRUE} (the default).


\code{prune_backups()} may delete files, depending on \code{max_backups}.
}

\section{Intervals}{


In \strong{rotor}, an interval is a character string in the form
\code{"<number> <interval>"}. The following intervals are possible:
\code{"day(s)"}, \code{"week(s)"}, \code{"month(s)"}, \code{"quarter(s)"}, \code{"year(s)"}.
The plural \code{"s"} is optional (so \code{"2 weeks"} and \code{"2 week"} are equivalent).
Please be aware that weeks are
\href{https://en.wikipedia.org/wiki/ISO_week_date}{ISOweeks}
and start on Monday (not Sunday as in some countries).

Interval strings can be used as arguments when backing up or rotating files,
or for pruning backup queues (i.e. limiting the number of backups of a
single) file.

When rotating/backing up \code{"1 months"} means "make a new backup if the last
backup is from the preceding month". E.g if the last backup of \code{myfile}
is from \code{2019-02-01} then \code{backup_time(myfile, age = "1 month")} will only
create a backup if the current date is at least \code{2019-03-01}.

When pruning/limiting backup queues, \code{"1 year"} means "keep at least most
one year worth of backups". So if you call
\code{backup_time(myfile, max_backups = "1 year")} on \code{2019-03-01}, it will create
a backup and then remove all backups of \code{myfile} before \code{2019-01-01}.
}

\examples{
# setup example file
tf <- tempfile("test", fileext = ".rds")
saveRDS(cars, tf)

# create two backups of `tf``
backup(tf)
backup(tf)
list_backups(tf)  # find all backups of a file

# If `size` is set, a backup is only created if the target file is at least
# that big. This is more useful for log rotation than for backups.
backup(tf, size = "100 mb")  # no backup becuase `tf` is to small
list_backups(tf)

# If `dry_run` is TRUE, backup() only shows what would happen without
# actually creating or deleting files
backup(tf, size = "0.1kb", dry_run = TRUE)

# rotate() is the same as backup(), but replaces `tf`` with an empty file
rotate(tf)
list_backups(tf)
file.size(tf)
file.size(list_backups(tf))

# prune_backups() can remove old backups
prune_backups(tf, 1)  # keep only one backup
list_backups(tf)

# rotate/backup_date() adds a date instead of an index
# you should not mix index backups and timestamp backups
# so we clean up first
prune_backups(tf, 0)
saveRDS(cars, tf)

# backup_date() adds the date instead of an index to the filename
backup_date(tf)

# `age` sets the minimum age of the last backup before creating a new one.
# the example below creates no new backup since it's less than a week
# since the last.
backup_date(tf, age = "1 week")

# `now` overrides the current date.
backup_date(tf, age = "1 year", now = "2999-12-31")
list_backups(tf)

# backup_time() creates backups with a full timestamp
backup_time(tf)

# It's okay to mix backup_date() and backup_time()
list_backups(tf)

# cleanup
prune_backups(tf, 0)
file.remove(tf)
}
\seealso{
\code{\link[=list_backups]{list_backups()}}
}
