\name{outreg}
\alias{outreg}
\title{Creates a publication quality result table for
regression models.}
\usage{
  outreg(modelList, title, label, modelLabels = NULL,
    varLabels = NULL, tight = TRUE, showAIC = FALSE,
    float = FALSE, request, runFuns, digits = 3,
    alpha = 0.05)
}
\arguments{
  \item{modelList}{A regression model or an R list of
  regression models.  If this is a named list, the names
  will be used as column labels, unless the argument
  modelLabels is supplied separately, which will override
  the names of modelList.}

  \item{title}{A title to be displayed on the top of the
  LaTeX regression table.}

  \item{label}{A string to be used as a LaTeX label in the
  table to be created.}

  \item{modelLabels}{A vector of charater string variables,
  one for each element in modelList. Will override the
  names in modelList.}

  \item{varLabels}{To beautify the parameter names printed.
  Must be a named vector in the format c(parmname =
  "displayName", parmname = "displayName"). Include as many
  parameters as desired, it is not necessary to supply new
  labels for all of the parameters. "displayName" must be
  valid laTeX. This feature is useful if your variable
  names have any illegal LaTeX characters, such as "$" or
  "_".}

  \item{tight}{If TRUE, parameter estimates and standard
  errors are printed in a single column.  If FALSE,
  parameter estimates and standard errors are printed side
  by side.}

  \item{showAIC}{If TRUE, the AIC estimate is included with
  the diagnostic values}

  \item{float}{Include boilerplate for a table float, with
  the tabular markup inside it.}

  \item{request}{Extra information to be retrieved from the
  summary(model) and displayed. This must be a vector of
  named arguments, such as c(adj.r.squared = "adj $R^2$",
  fstatistic = "F"). The name must be a valid name of the
  output object, the value should be the label the user
  wants printed in the table. See details.}

  \item{runFuns}{A list of functions}

  \item{digits}{Default = 3. How many digits after decimal
  sign are to be displayed.}

  \item{alpha}{Default = 0.05. I think stars are dumb, but
  enough people have asked me for more stars that I'm
  caving in. Enter c(0.05, 0.01, 0.001) to see what
  happens.}
}
\value{
  A character vector, one element per row of the regression
  table.
}
\description{
  This provides "markup" that the user is will copy into a
  LaTeX document. The table is a LaTeX tabular environment,
  which can be provided with the LaTeX boilerplate for a
  floating table if desired.  Set float = TRUE if that
  table boilerplate is needed. This can be called within
  Sweave documents. The markup generated by the basic usage
  will generally be presentable as is, while user requests
  for additional details may cause output that needs some
  hand-editing.
}
\details{
  In rockchalk 1.7.5, this is revised significantly. It now
  creates a character string variable, one element for each
  row. It writes that on the screen but also creates an
  output object.

  The table include a minimally sufficient (in my opinion)
  model summary.  For any fitted model, \code{outreg()}
  will present the parameter estimates and standard errors,
  and it will also scan the summary of the object for some
  summary values and goodness of fit indicators.  Some
  users may want more information. lm fits will not include
  output for the "fstatistic" that is reported in the lm's
  summary object, but users can ask for it with the
  argument \code{request}. Similarly, the argument
  \code{runFuns} can ask for additional diagnostic
  functions to be reported.

  While \code{outreg()} originally designed for models
  fitted by \code{lm()} and \code{glm()}, it will now work
  for any kind of regression model that has methods
  \code{summary()}, \code{coef()}, \code{vcov()} and
  \code{nobs()}.  This will not succeed, however, if the
  regression model's summary function does not create an
  object that holds values in a workable structure (in the
  same way that \code{lm()} and \code{glm()} do). Please
  pressure people who write R packages that claim to "do
  regression" to supply methods to do the basic work that
  we need to compile tables.

  Two types of customization arguments were introduced with
  rockchalk version 1.7.3.  The new arguments are
  \code{request} and \code{runFuns}. Beta test reports are
  welcome!

  The \code{request} argument supplies a list of names of
  summary output elements that are desired. The format is a
  pair, a value to be retrieved and a name to be printed
  for it. With the \code{lm()} regression, for example, one
  might want the output of the F test and the adjusted
  R-square.  An example usage might be \code{request =
  c(adj.r.squared = "adj. $R^2", fstatistic = "F")}. The
  names may be included in quotation marks if the user
  desires to do so: \code{request = c("adj.r.squared" =
  "adj. $R^2", "fstatistic" = "F")}.  The value on the left
  is the name of the desired information in the summary
  object, while the value on the right is \emph{any} valid
  LaTeX markup that the user desires to display in the
  first column of the table.  \code{request} terms that
  generate a single numerical value will generally work
  fine, while requests that ask for more structured
  information, such as the F test (including the 2 degrees
  of freedom values) are still a work in progress.

  The \code{runFuns} argument is inspired by a user
  request: could this include the BIC or other summaries
  that some models report? We have to run the BIC function,
  and divert the result into the correct column of the
  result table. Any R function, whether supplied with and R
  package or in the user's own code, may be used. This is a
  two-part specification, one representing the function to
  be run, the other representing the name that is desired
  in the output. For example, it might be \code{runFuns =
  c("AIC" = "Akaike Criterion", "BIC" = "Schwartz
  Criterion", "logLik" = "LL")}.
}
\note{
  There are many R packages that can be used to create
  LaTeX regression tables. memisc, texreg, apsrtable,
  xtables, and rms are some. This "outreg" version was in
  use in our labs before we were aware that those packages
  were in development. It is not intended as a competitor,
  it is just a slightly different version of the same that
  is more suited to our needs.
}
\examples{
set.seed(2134234)
dat <- data.frame(x1 = rnorm(100), x2 = rnorm(100))
dat$y1 <- 30 + 5 * rnorm(100) + 3 * dat$x1 + 4 * dat$x2
dat$y2 <- rnorm(100) + 5 * dat$x2
m1 <- lm(y1 ~ x1, data = dat)
m2 <- lm(y1 ~ x2, data = dat)
m3 <- lm(y1 ~ x1 + x2, data = dat)
gm1 <- glm(y1 ~ x1, family = Gamma, data = dat)
outreg(m1, title = "My One Tightly Printed Regression", float = TRUE )
outreg(m1, title = "My One Tightly Printed Regression", float = TRUE )
outreg(m1, tight = FALSE, modelLabels=c("Fingers"),
    title = "My Only Spread Out Regressions", float = TRUE,
    alpha = c(0.05, 0.01, 0.001))

outreg(list(ModelA = m1, "Model B label with Spaces" = m2),
    varLabels = list(x1 = "Billie"),
    title = "My Two Linear Regressions", request = c(fstatistic = "F"))

outreg(list(ModelA = m1, ModelB = m2),
    modelLabels = c("Overrides ModelA", "Overrides ModelB"),
    varLabels = list(x1 = "Billie"),
    title = "Note modelLabels Overrides model names")

outreg(list(m1, m2), modelLabels = c("Whatever", "Whichever"),
    title = "Still have showAIC argument, as in previous versions",
    showAIC = TRUE, float = TRUE)

outreg(list(m1, m2), modelLabels = c("Whatever", "Whichever"),
    title = "Another way to get AIC output",
    runFuns = c("AIC" = "Akaike IC"))

outreg(list("Amod" = m1, "Bmod" = m2, "Gmod" = m3),
    title = "My Three Linear Regressions", float = FALSE)

outreg(list(m1, m2, m3), tight = FALSE,
    modelLabels = c("I Love really long titles", "Hate Long", "Medium"),
    float = FALSE)

outreg(list(gm1), modelLabels = c("GLM"), float = TRUE)

outreg(list(m1, gm1), modelLabels = c("OLS", "GLM"), float = TRUE,
    alpha = c(0.05, 0.01))

outreg(list(OLS = m1, GLM = gm1), float = TRUE,
    request = c(fstatistic = "F"), runFuns = c("BIC" = "BIC"))

outreg(list(OLS = m1, GLM = gm1), float = TRUE,
    request = c(fstatistic = "F"), runFuns = c("BIC" = "BIC"),
    digits = 5, alpha = 0.01)

outreg(list("OLS 1" = m1, "OLS 2" = m2,  GLM = gm1), float = TRUE,
    request = c(fstatistic = "F"),
    runFuns = c("BIC" = "BIC", logLik = "ll"),
    digits = 5, alpha = c(0.05, 0.01, 0.001))

outreg(list(ModelA = gm1, "Model B label with Spaces" = m2),
    request = c(fstatistic = "F"),
    runFuns = c("BIC" = "Schwarz IC", "AIC" = "Akaike IC",
    "nobs" = "N Again?"))
}
\author{
  Paul E. Johnson \email{<pauljohn@ku.edu>}
}
\keyword{regression}

