% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chains.R
\name{sample_chain}
\alias{sample_chain}
\title{Sample a Markov chain}
\usage{
sample_chain(
  target_distribution,
  initial_state,
  n_warm_up_iteration,
  n_main_iteration,
  proposal = barker_proposal(),
  adapters = list(scale_adapter(), shape_adapter()),
  show_progress_bar = TRUE,
  trace_warm_up = FALSE
)
}
\arguments{
\item{target_distribution}{Target stationary distribution for chain. One of:
\itemize{
\item A one-sided formula specifying expression for log density of target
distribution which will be passed to
\code{\link[=target_distribution_from_log_density_formula]{target_distribution_from_log_density_formula()}} to construct functions
to evaluate log density and its gradient using \code{\link[=deriv]{deriv()}}.
\item A \code{bridgestan::StanModel} instance (requires \code{bridgestan} to be
installed) specifying target model and data. Will be passed to
\code{\link[=target_distribution_from_stan_model]{target_distribution_from_stan_model()}} using default values for optional
arguments - to override call \code{\link[=target_distribution_from_stan_model]{target_distribution_from_stan_model()}}
directly and pass the returned list as the \code{target_distribution} argument
here.
\item A list with named entries \code{log_density} and \code{gradient_log_density}
corresponding to respectively functions for evaluating the logarithm of
the (potentially unnormalized) density of the target distribution and its
gradient (only required for gradient-based proposals). As an alternative
to \code{gradient_log_density} an entry \code{value_and_gradient_log_density} may
instead be provided which is a function returning both the value and
gradient of the logarithm of the (unnormalized) density of the target
distribution as a list under the names \code{value} and \code{gradient}
respectively. The list may also contain a named entry \code{trace_function},
correspond to a function which given current chain state outputs a named
vector or list of variables to trace on each main (non-adaptive) chain
iteration. If a \code{trace_function} entry is not specified, then the default
behaviour is to trace the position component of the chain state along
with the log density of the target distribution.
}}

\item{initial_state}{Initial chain state. Either a vector specifying just
the position component of the chain state or a list output by \code{chain_state}
specifying the full chain state.}

\item{n_warm_up_iteration}{Number of warm-up (adaptive) chain iterations to
run.}

\item{n_main_iteration}{Number of main (non-adaptive) chain iterations to
run.}

\item{proposal}{Proposal distribution object. Defaults to Barker proposal,
that is the output of \code{\link[=barker_proposal]{barker_proposal()}}. Proposal objects are lists which
must minimally define entries \code{sample}, a function to generate sample from
proposal distribution given current chain state and \code{log_density_ratio}, a
function to compute log density ratio for proposal for a given pair of
current and proposed chain states. If adapters are being used to adaptively
tune the proposal scale and shape parameters, which is the default
behaviour of \code{sample_chain}, then additionally the list must also define
entries: \code{update} a function for updating parameters of proposal,
\code{parameters} a function for getting current proposal parameter values,
\code{default_target_accept_prob} a function for getting proposal specific
default target acceptance probability for scale adaptation and
\code{default_initial_scale} a function for getting proposal and dimension
dependent default initial value for scale parameter.}

\item{adapters}{List of adapters to tune proposal parameters during warm-up.
Defaults to using list with instances of \code{\link[=scale_adapter]{scale_adapter()}} and
\code{\link[=shape_adapter]{shape_adapter()}}, corresponding to respectively, adapting the scale to
coerce the average acceptance rate to a target value using a dual-averaging
algorithm, and adapting the shape to an estimate of the covariance of the
target distribution.}

\item{show_progress_bar}{Whether to show progress bars during sampling.
Requires \code{progress} package to be installed to have an effect.}

\item{trace_warm_up}{Whether to record chain traces and adaptation /
transition statistics during (adaptive) warm-up iterations in addition to
(non-adaptive) main chain iterations.}
}
\value{
A list with entries
\itemize{
\item \code{final_state}: the final chain state,
\item \code{traces}: a matrix with named columns contained traced variables for each
main chain iteration, with variables along columns and iterations along
rows.
\item \code{statistics}: a matrix with named columns containing transition statistics
for each main chain iteration, with statistics along columns and iterations
along rows.
\item \code{warm_up_traces}: a matrix with named columns contained traced variables
for each warm-up chain iteration, with variables along columns and
iterations along rows. Only present if \code{trace_warm_up = TRUE}.
\item \code{warm_up_statistics}: a matrix with named columns containing adaptation and
transition statistics for each warm-up chain iteration, with statistics
along columns and iterations along rows. Only present if
\code{trace_warm_up = TRUE}.
}
}
\description{
Sample a Markov chain using Metropolis-Hastings kernel with a user-specified
target distribution and proposal (defaulting to Barker proposal), optionally
adapting proposal parameters in a warm-up stage.
}
\examples{
target_distribution <- list(
  log_density = function(x) -sum(x^2) / 2,
  gradient_log_density = function(x) -x
)
withr::with_seed(876287L, {
  results <- sample_chain(
    target_distribution,
    initial_state = stats::rnorm(2),
    n_warm_up_iteration = 1000,
    n_main_iteration = 1000
  )
})
}
