% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/angle.R
\name{angle}
\alias{angle}
\title{Calculate the angle to an origin}
\usage{
angle(
  data,
  x_col = NULL,
  y_col = NULL,
  origin = NULL,
  origin_fn = NULL,
  degrees_col_name = ".degrees",
  origin_col_name = ".origin",
  overwrite = FALSE
)
}
\arguments{
\item{data}{\code{data.frame} or \code{vector}.}

\item{x_col}{Name of x column in \code{`data`}. If \code{NULL} and \code{`data`} is a \code{vector},
the index of \code{`data`} is used. If \code{`data`} is a \code{data.frame}, it must be specified.}

\item{y_col}{Name of y column in \code{`data`}. If \code{`data`} is a \code{data.frame}, it must be specified.}

\item{origin}{Coordinates of the origin to calculate angle to.
A scalar to use in all dimensions
or a \code{vector} with one scalar per dimension.

\strong{N.B.} Ignored when \code{`origin_fn`} is not \code{NULL}.}

\item{origin_fn}{Function for finding the origin coordinates.

\strong{Input}: Each column will be passed as a \code{vector} in the order of \code{`cols`}.

\strong{Output}: A \code{vector} with one scalar per dimension.

Can be created with \code{\link[rearrr:create_origin_fn]{create_origin_fn()}} if you want to apply
the same function to each dimension.

E.g. \code{`create_origin_fn(median)`} would find the median of each column.

\strong{Built-in functions} are \code{\link[rearrr:centroid]{centroid()}},
\code{\link[rearrr:most_centered]{most_centered()}},
and \code{\link[rearrr:midrange]{midrange()}}}

\item{degrees_col_name}{Name of new column with the degrees.}

\item{origin_col_name}{Name of new column with the origin coordinates. If \code{NULL}, no column is added.}

\item{overwrite}{Whether to allow overwriting of existing columns. (Logical)}
}
\value{
\code{data.frame} (\code{tibble}) with the additional columns (degrees and origin coordinates).
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Calculates the angle between each data point \eqn{(x2, y2)} and the origin \eqn{(x1, y1)} with:
\deqn{atan2(y2 - y1, x2 - x1)}

And converts to degrees \code{[0-360)}, measured counterclockwise from the \code{\{x > x1, y = y1\}} line.

\ifelse{html}{\out{<img src='figures/angle_wheel.jpg' width="120" alt='Angles wheel'>}}{}

The origin can be supplied as coordinates or as a function that returns coordinates. The
latter can be useful when supplying a grouped \code{data.frame} and finding the angle to e.g. the centroid
of each group.
}
\examples{
# Attach packages
library(rearrr)
library(dplyr)
has_ggplot <- require(ggplot2)  # Attach if installed

# Set seed
set.seed(1)

# Create a data frame
df <- data.frame(
  "x" = runif(20),
  "y" = runif(20),
  "g" = rep(1:4, each = 5)
)

# Calculate angles in the two dimensions (x and y)
# With the origin at x=0.5, y=0.5
df_angles <- angle(
  data = df,
  x_col = "x",
  y_col = "y",
  origin = c(0.5, 0.5)
)
df_angles

# Plot points with degrees
# Degrees are measured counterclockwise around the
# positive side of the x-axis
if (has_ggplot){
  df_angles \%>\%
    ggplot(aes(x = x, y = y, color = .degrees)) +
    geom_segment(aes(x = 0.5, xend = 1, y = 0.5, yend = 0.5), color = "magenta") +
    geom_point() +
    theme_minimal()
}

# Calculate angles to the centroid for each group in 'g'
angle(
  data = dplyr::group_by(df, g),
  x_col = "x",
  y_col = "y",
  origin_fn = centroid
)
}
\seealso{
Other measuring functions: 
\code{\link{distance}()},
\code{\link{vector_length}()}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{measuring functions}
