#' @title Bland Altman Plots
#'
#' @description Construct and plot a Bland Altman plot in ggplot2.
#'
#' @details
#' Providing a \code{data.frame} with two columns, the function returns a ggplot
#' version of a Bland Altman plot with the specified confidence intervals.
#'
#' Two ways to call the plotting function.  If you submit a \code{data.frame}
#' \code{qblandaltman} then the data needed to produce the Bland Altman plot is
#' automatically generated by a call to \code{qblandaltman_build_data_frame}.
#' Alternatively, you may call \code{qblandaltman_build_data_frame} directly and
#' then call \code{qblandaltman}.  This might be helpful if you are putting
#' multiple Bland Altman plots together into one ggplot object.  See Examples.
#'
#' @param .data a \code{data.frame} with two columns.  If a \code{data.frame}
#' with more than two columns is used only the first two columns will be used.
#' @param alpha (Defaults to 0.05) place (1 - alpha)*100% confidence levels to
#' place on the plot. 
#' @param generate_data logical, defaults to TRUE.  If TRUE, then the call to
#' \code{qblandaltman_build_data_frame} is done automatically for you.  If
#' FALSE, then you should explicitly call \code{qblandaltman_build_data_frame}
#' before calling \code{qblandaltman}.
#'
#' @return a ggplot.  Minimula aesthetics have been used so that the user may
#' modify the graphic as desired with ease.
#'
#' @examples
#' \dontrun{
#' # load ggplot2 and the diamonds data set
#' library(ggplot2)
#' data(diamonds, package = "ggplot2")
#'
#' # compare a simple regression to random noise
#' dat <- 
#'   data.frame(fitted(lm(price ~ poly(carat, 4), data = diamonds)),  # fitted values
#'              diamonds$price + rnorm(nrow(diamonds), sd = 0.2),     # observed with noise
#'              pi)                                                   # extra column
#' qblandaltman(dat)
#' 
#' # simple example
#' dat <- data.frame(eval1 = rpois(100, 3), eval2 = rpois(100, 3.4)) 
#' qblandaltman(dat)
#'
#' last_plot() + theme_bw()
#' 
#' # Two plots in one ggplot object
#' set.seed(42)
#' dat1 <- data.frame(eval1 = rnorm(100), eval2 = rt(100, df = 1))
#' dat2 <- data.frame(eval1 = rpois(50, 3), eval2 = rpois(50, 4))
#' 
#' # individual plots
#' qblandaltman(dat1)
#' qblandaltman(dat2)
#' 
#' # combined plots
#' dat <- rbind(cbind(set = "rnorm", qblandaltman_build_data_frame(dat1)), 
#'              cbind(set = "rpois", qblandaltman_build_data_frame(dat2)))
#' qblandaltman(dat, generate_data = FALSE) + facet_wrap( ~ set)
#' }
#' @import ggplot2 dplyr
#' @export   
#' @rdname qblandaltman
qblandaltman <- function(.data, alpha = getOption("qwraps2_alpha", 0.05), generate_data = TRUE) { 

  if (is.null(attr(.data, "qwraps2_generated"))) { 
    if (generate_data) {
      .data <- qblandaltman_build_data_frame(.data, alpha)
    }
  }

  ggplot2::ggplot(.data) + 
  ggplot2::aes_string(x = 'avg', y = 'diff') + 
  ggplot2::geom_point() + 
  ggplot2::geom_hline(ggplot2::aes_string(yintercept = 'lcl'), lty = 2) + 
  ggplot2::geom_hline(ggplot2::aes_string(yintercept = 'ucl'), lty = 2) + 
  ggplot2::geom_hline(ggplot2::aes_string(yintercept = 'mean_diff'), lty = 3) 
}

#' @export
#' @rdname qblandaltman
qblandaltman_build_data_frame <- function(.data, alpha = getOption("qwraps2_alpha", 0.05)) { 
  rtn <-
    dplyr::mutate_(data.frame(x1 = .data[[1]], x2 = .data[[2]]),
                   avg  = ~ (x1 + x2) / 2, 
                   diff = ~ (x2 - x1), 
                   mean_diff = "mean(diff)", 
                   sd_diff   = "sd(diff)", 
                   lcl       = ~ mean_diff + qnorm(alpha / 2) * sd_diff,
                   ucl       = ~ mean_diff + qnorm(1 - alpha / 2) * sd_diff) 
  rtn <- tbl_df(rtn)

  attr(rtn, "qwraps2_generated") = TRUE

  return(rtn)
}

