# .onLoad <- function(libname, pkgname) {
#   jarpath = system.file("java", "auc2.jar", package = "prcbench")
#   .jpackage(pkgname, lib.loc = libname, morePaths = jarpath)
# }

#' Base class of performance evaluation tools
#'
#' \code{ToolIFBase} is an abstract class to provide a uniform interface for
#' performance evaluation tools.
#'
#' @section Methods:
#' \itemize{
#'  \item \code{call(testset, calc_auc, store_res)}: It calls an actual
#'      tool to calculate Precision-Recall curves.
#'      \describe{
#'        \item{\code{testset}}{\code{R6} object generated by the
#'                               \code{create_testset} function.}
#'        \item{\code{calc_auc}}{A Boolean value to specify whether the AUC
#'                               score should be calculated.}
#'        \item{\code{store_res}}{A Boolean value to specify whether the
#'                                calculated curve is retrieved and stored.}
#'       }
#'  \item \code{get_toolname()}: Get the name of the tool.
#'  \item \code{set_toolname(toolname)}: Set the name of the tool.
#'  \item \code{get_setname()}: Get the name of the tool set.
#'  \item \code{set_setname(setname)}: Set the name of the tool set.
#'  \item \code{get_result()}: Get a list with curve values and the AUC score.
#'  \item \code{get_x()}: Get calculated recall values.
#'  \item \code{get_y()}: Get calculated precision values.
#'  \item \code{get_auc()}: Get tne AUC score.
#' }.
#'
#' @seealso \code{\link{ToolROCR}}, \code{\link{ToolAUCCalculator}},
#'   \code{\link{ToolPerfMeas}}, \code{\link{ToolPRROC}},
#'   and \code{\link{Toolprecrec}} are derived from this class.
#'   \code{\link{create_toolset}} for creating a list of tools.
#'
#' @docType class
#' @format An R6 class object
#' @export
ToolIFBase <- R6::R6Class(
  "ToolIFBase",
  public = list(
    initialize = function(...) {
      private$set_def_params(...)
    },
    call = function(testset, calc_auc, store_res) {
      if (missing(calc_auc)) {
        calc_auc <- private$def_calc_auc
      }
      if (missing(store_res)) {
        store_res <- private$def_store_res
      }
      if (is.function(private$f_wrapper)) {
        result <- private$f_wrapper(testset, calc_auc, store_res)
        if (store_res && !is.null(result)) {
          private$set_result(result)
        } else if (calc_auc && !is.null(result$auc)) {
          private$set_auc(result$auc)
        }
      }
      private$called <- TRUE
      self
    },
    get_toolname = function() {private$toolname},
    set_toolname = function(toolname) {private$toolname <- toolname},
    get_setname = function() {private$setname},
    set_setname = function(setname) {private$setname <- setname},
    get_result = function() {private$result},
    get_x = function() {private$result[["x"]]},
    get_y = function() {private$result[["y"]]},
    get_auc = function() {private$result[["auc"]]},
    print = function(...) {
      cat("\n")
      cat("    === Tool interface ===\n")
      cat("\n")

      cat("    Tool name:           ", self$get_toolname(), "\n")
      cat("    Calculate AUC score: ")
      if (private$def_calc_auc) {
        cat(" Yes\n")
      } else {
        cat(" No\n")
      }
      cat("    Store results:       ")
      if (private$def_store_res) {
        cat(" Yes\n")
      } else {
        cat(" No\n")
      }
      cat("    Prediction performed:")
      if (private$called) {
        cat(" Yes\n")
      } else {
        cat(" No\n")
      }
      cat("    Available methods:    call(testset, calc_auc, store_res)\n")
      cat("                          get_toolname()\n")
      cat("                          set_toolname(toolname)\n")
      cat("                          get_setname()\n")
      cat("                          set_setname(setname)\n")
      cat("                          get_result()\n")
      cat("                          get_x()\n")
      cat("                          get_y()\n")
      cat("                          get_auc()\n")
      private$print_methods()
      if (private$helpfile) {
        cat("    Help file:           ",
            paste0("help(\"", class(self)[1], "\")"))
      }
      cat("\n")
      invisible(self)
    }
  ),
  private = list(
    set_def_params = function(...) {
      arglist <- list(...)
      if (length(arglist) > 0) {
        if ("setname" %in% names(arglist)){
          private$setname <- arglist[["setname"]]
        }
        if ("calc_auc" %in% names(arglist)){
          private$def_calc_auc <- arglist[["calc_auc"]]
        }
        if ("store_res" %in% names(arglist)){
          private$def_store_res <- arglist[["store_res"]]
        }
        if ("x" %in% names(arglist)){
          private$result$x <- arglist[["x"]]
        }
        if ("y" %in% names(arglist)){
          private$result$y <- arglist[["y"]]
        }
      }
    },
    toolname = NA,
    setname = NA,
    called = FALSE,
    def_calc_auc = TRUE,
    def_store_res = TRUE,
    print_methods = function() {invisible(NULL)},
    set_result = function(val) {private$result <- val},
    set_auc = function(val) {private$result[["auc"]] <- val},
    result = list(x = NA, y = NA, auc = NA),
    f_wrapper = function(testset, calc_auc, store_res) {NULL},
    helpfile = TRUE
  )
)

#' R6 class of the ROCR tool
#'
#' \code{ToolROCR} is a wrapper class for
#' the \href{https://rocr.bioinf.mpi-sb.mpg.de/}{ROCR} tool, which is an R
#' library that provides calculations of various performance evaluation
#' measures.
#'
#' @section Inheritance:
#' \code{\link{ToolIFBase}}
#'
#' @section Methods:
#' Following nine methods are inherited from \code{\link{ToolIFBase}}. See
#' \code{\link{ToolIFBase}} for the method descriptions.
#' \itemize{
#'   \item \code{call(testset, calc_auc, store_res)}
#'   \item \code{get_toolname()}
#'   \item \code{set_toolname(toolname)}
#'   \item \code{get_setname()}
#'   \item \code{set_setname(setname)}
#'   \item \code{get_result()}
#'   \item \code{get_x()}
#'   \item \code{get_y()}
#'   \item \code{get_auc()}
#' }
#'
#' @seealso This class is derived from \code{\link{ToolIFBase}}.
#'    \code{\link{create_toolset}} for creating a list of tools.
#'
#' @examples
#' ## Initialization
#' toolrocr <- ToolROCR$new()
#'
#' ## Show object info
#' toolrocr
#'
#' ## create_toolset should be used for benchmarking and curve evaluation
#' toolrocr2 <- create_toolset("ROCR")
#'
#' @docType class
#' @format An R6 class object.
#' @export
ToolROCR <- R6::R6Class(
  "ToolROCR", inherit = ToolIFBase,
  private = list(toolname = "ROCR", f_wrapper = .rocr_wrapper)
)

#' R6 class of the AUCCalculator tool
#'
#' \code{ToolAUCCalculator} is a wrapper class for
#' the \href{http://mark.goadrich.com/programs/AUC/}{AUCCalculator} tool, which
#' is a Java library that provides calculations of ROC and Precision-Recall
#' curves.
#'
#' @section Inheritance:
#' \code{\link{ToolIFBase}}
#'
#' @section Methods:
#'
#' \describe{
#'   \item{\code{set_jarpath(jarpath)}}{
#'     It sets an AUCCalculator jar file.
#'
#'     \describe{
#'       \item{\code{jarpath}}{File path of the AUCCalculator jar file,
#'                             e.g. \code{"/path1/path2/auc2.jar"}.}
#'     }
#'   }
#' }
#'
#' Following nine methods are inherited from \code{\link{ToolIFBase}}. See
#' \code{\link{ToolIFBase}} for the method descriptions.
#' \itemize{
#'   \item \code{call((testset, calc_auc, store_res)}
#'   \item \code{get_toolname()}
#'   \item \code{set_toolname(toolname)}
#'   \item \code{get_setname()}
#'   \item \code{set_setname(setname)}
#'   \item \code{get_result()}
#'   \item \code{get_x()}
#'   \item \code{get_y()}
#'   \item \code{get_auc()}
#' }
#'
#' @seealso This class is derived from \code{\link{ToolIFBase}}.
#'    \code{\link{create_toolset}} for creating a list of tools.
#'
#' @examples
#' ## Initialization
#' toolauccalc <- ToolAUCCalculator$new()
#'
#' ## Show object info
#' toolauccalc
#'
#' ## create_toolset should be used for benchmarking and curve evaluation
#' toolauccalc2 <- create_toolset("AUCCalculator")
#'
#' @docType class
#' @format An R6 class object.
#' @export
ToolAUCCalculator <- R6::R6Class(
  "ToolAUCCalculator", inherit = ToolIFBase,
  public = list(
    initialize = function(...) {
      private$set_def_params(...)

      arglist <- list(...)
      if (length(arglist) > 0) {
        if ("jarpath" %in% names(arglist)){
          private$jarpath <- arglist[["jarpath"]]
        }
      }
      private$f_setjar()
    },
    set_jarpath = function(jarpath = NULL) {
      private$jarpath <- jarpath
      private$f_setjar()
    },
    set_curvetype = function(curvetype = "SPR") {
      if (assertthat::assert_that(assertthat::is.string(curvetype))
          && (toupper(curvetype) %in% c("SPR", "PR", "ROC"))) {
        private$curvetype <- toupper(curvetype)
        private$f_setjar()
      }
    },
    set_auctype = function(auctype) {
      if (assertthat::assert_that(assertthat::is.string(auctype))
          && (tolower(auctype) %in% c("java", "r"))) {
        private$auctype = tolower(auctype)
      }
    }
  ),
  private = list(
    toolname = "AUCCalculator",
    curvetype = "SPR",
    auctype = "java",
    print_methods = function() {
      cat("                          set_jarpath(jarpath)\n")
      cat("                          set_curvetype(curvetype)\n")
      cat("                          set_auctype(auctype)\n")
    },
    auc2 = NA,
    jarpath = NA,
    f_setjar = function() {
      if (is.na(private$jarpath)) {
        jarpath <- system.file("java", "auc2.jar", package = "prcbench")
      } else {
        jarpath <- private$jarpath
      }
      private$auc2 <- .get_java_obj("auc2", jarpath, private$curvetype)
    },
    f_wrapper = function(testset, calc_auc, store_res) {
      .auccalc_wrapper(testset, private$auc2, calc_auc, store_res,
                       private$auctype)
    }
  )
)

#' R6 class of the PerfMeas tool
#'
#' \code{ToolPerfMeas} is a wrapper class for
#' the \href{https://cran.r-project.org/package=PerfMeas}{PerfMeas} tool,
#' which is an R library that provides several performance measures.
#'
#' @section Inheritance:
#' \code{\link{ToolIFBase}}
#'
#' @section Methods:
#' Following nine methods are inherited from \code{\link{ToolIFBase}}. See
#' \code{\link{ToolIFBase}} for the method descriptions.
#' \itemize{
#'   \item \code{call(testset, calc_auc, store_res)}
#'   \item \code{get_toolname()}
#'   \item \code{set_toolname(toolname)}
#'   \item \code{get_setname()}
#'   \item \code{set_setname(setname)}
#'   \item \code{get_result()}
#'   \item \code{get_x()}
#'   \item \code{get_y()}
#'   \item \code{get_auc()}
#' }
#'
#' @seealso This class is derived from \code{\link{ToolIFBase}}.
#'    \code{\link{create_toolset}} for creating a list of tools.
#'
#' @examples
#' ## Initialization
#' toolperf <- ToolPerfMeas$new()
#'
#' ## Show object info
#' toolperf
#'
#' ## create_toolset should be used for benchmarking and curve evaluation
#' toolperf2 <- create_toolset("PerfMeas")
#'
#' @docType class
#' @format An R6 class object.
#' @export
ToolPerfMeas <- R6::R6Class(
  "ToolPerfMeas", inherit = ToolIFBase,
  private = list(
    toolname = "PerfMeas",
    available = TRUE,
    f_wrapper = function(testset, calc_auc, store_res) {
      if (private$available && !requireNamespace("PerfMeas", quietly = TRUE)) {
        print("PerfMeas is not available.")
        private$available = FALSE
      }
      if (private$available) {
        .pm_wrapper(testset, calc_auc, store_res)
      } else {
        if (store_res) {
          x <- seq(0.0, 1.0, 0.1)
          y <- rep(0.5, length(x))
          list(x = x, y = y, auc = 0.5)
        } else {
          NULL
        }
      }
    }
  )
)

#' R6 class of the PRROC tool
#'
#' \code{ToolPRROC} is a wrapper class for
#' the \href{https://cran.r-project.org/package=PRROC}{PRROC} tool, which
#' is an R library that provides calculations of ROC and Precision-Recall
#' curves.
#'
#' @section Inheritance:
#' \code{\link{ToolIFBase}}
#'
#' @section Methods:
#' \describe{
#'   \item{\code{set_curve(val)}}{A Boolean value to specify whether a
#'                                Precision-Recall curve is calculated.}
#'   \item{\code{set_minStepSize(val)}}{A numeric value to specify the minimum
#'                                      step size between two intermediate
#'                                      points.}
#' }
#'
#' Following nine methods are inherited from \code{\link{ToolIFBase}}. See
#' \code{\link{ToolIFBase}} for the method descriptions.
#' \itemize{
#'   \item \code{call(testset, calc_auc, store_res)}
#'   \item \code{get_toolname()}
#'   \item \code{set_toolname(toolname)}
#'   \item \code{get_setname()}
#'   \item \code{set_setname(setname)}
#'   \item \code{get_result()}
#'   \item \code{get_x()}
#'   \item \code{get_y()}
#'   \item \code{get_auc()}
#' }
#'
#' @seealso This class is derived from \code{\link{ToolIFBase}}.
#'    \code{\link{create_toolset}} for creating a list of tools.
#'
#' @examples
#' ## Initialization
#' toolprroc <- ToolPRROC$new()
#'
#' ## Show object info
#' toolprroc
#'
#' ## create_toolset should be used for benchmarking and curve evaluation
#' toolprroc2 <- create_toolset("PRROC")
#'
#' @docType class
#' @format An R6 class object.
#' @export
ToolPRROC <- R6::R6Class(
  "ToolPRROC", inherit = ToolIFBase,
  public = list(
    initialize = function(...) {
      private$set_def_params(...)

      arglist <- list(...)
      if (length(arglist) > 0) {
        if ("curve" %in% names(arglist)){
          private$curve <- arglist[["curve"]]
        }
        if ("minStepSize" %in% names(arglist)){
          private$minStepSize <- arglist[["minStepSize"]]
        }
        if ("aucType" %in% names(arglist)){
          private$aucType <- arglist[["aucType"]]
        }
      }
    },
    set_curve = function(val) {private$curve <- val},
    set_minStepSize = function(val) {private$minStepSize <- val},
    set_aucType = function(val) {private$aucType <- val}
  ),
  private = list(
    toolname = "PRROC",
    print_methods = function() {
      cat("                          set_curve(val)\n")
      cat("                          set_minStepSize(val)\n")
      cat("                          set_aucType(val)\n")
    },
    f_wrapper = function(testset, calc_auc, store_res) {
      .prroc_wrapper(testset, calc_auc, store_res, private$curve,
                     private$minStepSize, private$aucType)
    },
    curve = TRUE,
    minStepSize = 0.01,
    aucType = 1
  )
)

#' R6 class of the precrec tool
#'
#' \code{Toolprecrec} is a wrapper class for
#' the \href{https://cran.r-project.org/package=precrec}{precrec} tool,
#' which is an R library that provides calculations of ROC and Precision-Recall
#' curves.
#'
#' @section Inheritance:
#' \code{\link{ToolIFBase}}
#'
#' @section Methods:
#' Following nine methods are inherited from \code{\link{ToolIFBase}}. See
#' \code{\link{ToolIFBase}} for the method descriptions.
#' \itemize{
#'   \item \code{call(testset, calc_auc, store_res)}
#'   \item \code{get_toolname()}
#'   \item \code{set_toolname(toolname)}
#'   \item \code{get_setname()}
#'   \item \code{set_setname(setname)}
#'   \item \code{get_result()}
#'   \item \code{get_x()}
#'   \item \code{get_y()}
#'   \item \code{get_auc()}
#' }
#'
#' @seealso This class is derived from \code{\link{ToolIFBase}}.
#'    \code{\link{create_toolset}} for creating a list of tools.
#'
#' @examples
#' ## Initialization
#' toolprecrec <- Toolprecrec$new()
#'
#' ## Show object info
#' toolprecrec
#'
#' ## create_toolset should be used for benchmarking and curve evaluation
#' toolprecrec2 <- create_toolset("precrec")
#'
#' @docType class
#' @format An R6 class object.
#' @export
Toolprecrec <- R6::R6Class(
  "Toolprecrec", inherit = ToolIFBase,
  public = list(
    initialize = function(...) {
      private$set_def_params(...)

      arglist <- list(...)
      if (length(arglist) > 0) {
        if ("x_bins" %in% names(arglist)){
          private$x_bins <- arglist[["x_bins"]]
        }
      }
    },
    set_x_bins = function(x_bins) {private$x_bins <- x_bins}
  ),
  private = list(
    toolname = "precrec",
    print_methods = function() {
      cat("                          set_x_bins(x_bins)\n")
    },
    f_wrapper = function(testset, calc_auc, store_res) {
      .precrec_wrapper(testset, calc_auc, store_res, private$x_bins)
    },
    x_bins = 1000
  )
)
