% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geoMite.R
\docType{data}
\name{geoMite}
\alias{geoMite}
\title{Borcard's Oribatid Mite Data Set - Geographic Information System Version -}
\format{
A list with five \code{\link[sf]{sf}} data frames:
\describe{
\item{core}{ A data frame with 70 rows (peat cores) containing point
geometries and 46 fields containing values of environmental variables at the
locations of the cores as well as the number of individuals of one of 35
Oribatid species observed in the cores (see details). }
\item{water}{ A data frame with three rows containing polygon geometries and
a single field: a \code{\link{factor}} variable named "Type" and specifying
whether the polygon represents open water (value == "Water") or flooded areas
(value == "Flooded") at the time of sampling. }
\item{substrate}{ A data frame with 13 rows containing polygon geometries
and seven fields. The first field is a \code{\link{factor}} that specifies
one of six substrate classes (see details) and the remaining six fields are
binary variables for each of these six substrate classes that take the value
1 when the polygon is of the class being represented by the that variable and
otherwise take the value 0. }
\item{shrub}{ A data frame with four rows containing polygon geometries and
a single field: an \code{\link{ordered}} variable named "Type" and specifying
whether the polygon represents areas with no shrub (value == "None"), a few
shrubs (value == "Few"), or many shrubs (value == "Many"). }
\item{topo}{ A data frame with four rows containing polygon geometries and a
single field: a \code{\link{factor}} variable named "Type" and specifying the
type of peat micro-topography. There are two such types: "Blanket" (flat
area) and "Hummock" (raised bumps). }
}
}
\usage{
data(geoMite)
}
\description{
Oribatid mite community data in a peat bog surrounding Lac Geai, QC, Canada
}
\details{
Fields of the point geometry (\code{geoMite$core}) are:
\describe{
\item{SubsDens}{ Substrate density (g/L). }
\item{WatrCont}{ Water content of the peat (g/L) }
\item{Substrate-prefixed}{ Six binary variables describing the substrate(s)
from which the peat core samples were collected. Further details are given
below. }
\item{Shrub}{ A three-level \code{\link{ordered}} factor describing the
presence and abundance of shrubs (mainly Ericaceae ) on the peat surface. }
\item{Topo}{ A two-level factor describing the microtopography of the peat
mat. }
\item{Flooded}{ A binary variable specifying whether the area in which the
core was sampled was flooded at the time of sampling. }
\item{Species-prefixed}{ Counts of one of 35 Oribatid species identified
purely on the basis of their morphology. }
}

The types of substrates are described as follows:
\describe{
\item{Sphagn1}{ Sphagnum magellanicum (with a majority of S. rubellum). }
\item{Sphagn2}{ Sphagnum rubellum. }
\item{Sphagn3}{ Sphagnum nemoreum (with a minority of S. angustifolium). }
\item{Sphagn4}{ Sphagnum rubellum and S. magellanicum in equal parts. }
\item{Litter}{ Ligneous litter. }
\item{Barepeat}{ Bare peat. }
}
These types are not mutually exclusive categories: cores were sometimes taken
at the boundary between two or more substrate types and thus belong to many
of these categories.

As stated earlier, identification of the Oribatid species was carried out
solely on the basis of their morphology as little is known on the ecology of
these small animals.

Geometries in geoMite$water, geoMite$substrate, geoMite$shrub, and
geoMite$topo were generated by outlining images from Fig. 1 in Borcard et al.
(1994) using a square grid with a resolution of about 10 mm. Because of the
inaccuracy to the available printed document the actual resolution is
probably inferior (i.e., 10 cm in both the x and y direction).

Orientation: the X coordinates corresponds to distances going from the edge
of the water to the edge of the forest. The Y coordinates correspond the
distances along the lake's shore.
}
\examples{
data(geoMite)

attach(geoMite)

## Color definitions:
col <- list()
col[["substrate"]] <- c(Sphagn1 = "#00ff00", Sphagn2 = "#fffb00",
                        Sphagn3 = "#774b00", Sphagn4 = "#ff8400",
                        Litter = "#ee00ff", Barepeat = "#ff0004")
col[["water"]] <- c(Water = "#008cff", Flooded = "#ffffff00",
                    core = "#000000ff")
col[["shrub"]] <- c(None = "#dfdfdf", Few = "#a7a7a7", Many = "#5c5c5c")
col[["topo"]] <- c(Blanket = "#74cd00", Hummock = "#bc9d00")

## Graphical paramters:
p <- par(no.readonly = TRUE)
par(mar=c(0,0,1,0), mfrow=c(1L,4L))

## Substrate:
plot(st_geometry(substrate), col=col[["substrate"]][substrate$Type],
     main="Substrate")
plot(st_geometry(water[1L,]), col=col[["water"]][water[1L,]$Type], add=TRUE)
plot(st_geometry(water[-1L,]), col=col[["water"]][water[-1L,]$Type], lty=3L,
     add=TRUE)
plot(st_geometry(core), pch = 21L, bg = "black", add=TRUE)

## Shrubs:
plot(st_geometry(shrub), col = col[["shrub"]][shrub$Type], main="Shrubs")
plot(st_geometry(water[1L,]), col=col[["water"]][water[1L,]$Type], add=TRUE)
plot(st_geometry(water[-1L,]), col=col[["water"]][water[-1L,]$Type], lty=3L,
     add=TRUE)
plot(st_geometry(core), pch = 21L, bg = "black", add=TRUE)

## Topograghy:
plot(st_geometry(topo), col = col[["topo"]][topo$Type], main="Topography")
plot(st_geometry(water[1L,]), col=col[["water"]][water[1L,]$Type], add=TRUE)
plot(st_geometry(water[-1L,]), col=col[["water"]][water[-1L,]$Type], lty=3L,
     add=TRUE)
plot(st_geometry(core), pch = 21L, bg = "black", add=TRUE)

## Legends:
plot(NA, xlim=c(0,1), ylim=c(0,1), axes = FALSE)
legend(x=0, y=0.9, pch=22L, pt.cex = 2.5, pt.bg=col[["substrate"]],
       box.lwd = 0, legend=names(col[["substrate"]]), title="Substrate")
legend(x=-0.025, y=0.6, pch=c(22L,NA,21L), pt.cex = c(2.5,NA,1),
       pt.bg=col[["water"]], box.lwd = 0, lty = c(0L,3L,NA),
       legend=c("Open water","Flooded area","Peat core"))
legend(x=0, y=0.4, pch=22L, pt.cex = 2.5, pt.bg=col[["shrub"]], box.lwd = 0,
       legend=names(col[["shrub"]]), title="Shrubs")
legend(x=0, y=0.2, pch=22L, pt.cex = 2.5, pt.bg=col[["topo"]], box.lwd = 0,
       legend=names(col[["topo"]]), title="Topography")

### Display the species counts

## Get the species names:
unlist(
  lapply(
    strsplit(colnames(core),".",fixed=TRUE),
    function(x) if(x[1L] == "Species") x[2L] else NULL
  )
) -> spnms

## See the maximum counts for all the species
apply(st_drop_geometry(core[,paste("Species",spnms,sep=".")]),2L,max)

## Species selection to display:
sel <- c("Brachysp","Hoplcfpa","Oppinova","Limncfci","Limncfru")

## Range of counts to display:
rng <- log1p(c(0,1000))

colmap <- grey(seq(1,0,length.out=256L))

## Update the graphical parameters for this example
par(mar=c(0,0,2,0), mfrow=c(1L,length(sel) + 1L))

## Display each species in the selection over the substrate map
for(sp in sel) {
  plot(st_geometry(substrate), col=col[["substrate"]][substrate$Type],
       main=sp)
  plot(st_geometry(core), pch=21L, add = TRUE, cex=1.5,
       bg=colmap[1 + 255*log1p(core[[paste("Species",sp,sep=".")]])/rng[2L]])
}

## Display the colour chart for the species counts:
par(mar=c(2,7,3,1))
image(z=matrix(seq(0,log1p(1000),length.out=256L),1L,256L), col=colmap,
      xaxt="n", yaxt="n", y=seq(0,log1p(1000),length.out=256L), xlab="",
      cex.lab = 1.5,
      ylab=expression(paste("Counts by species (",ind~core^{-1},")")))
axis(2L, at=log1p(c(0,1,3,10,30,100,300,1000)), cex.axis = 1.5,
     label=c(0,1,3,10,30,100,300,1000))

## Restore graphical parameters:
par(p)

}
\references{
Borcard, D. and Legendre, P. 1994. Environmental Control and Spatial
Structure in Ecological Communities: An Example Using Oribatid Mites
(Acari, Oribatei). Environ. Ecol. Stat. 1(1): 37-61 \doi{10.1007/BF00714196}

Borcard, D., Legendre, P., and Drapeau, P. 1992. Partialling out the spatial
component of ecological variation. Ecology, 73, 1045-1055.
\doi{10.2307/1940179}

Borcard, D.; Legendre, P.; and Gillet, F. 2018. Numerical Ecology with R
(2nd Edition) Sprigner, Cham, Switzerland. \doi{10.1007/978-3-319-71404-2}
}
\seealso{
Data set \code{oribatid} from package \code{ade4}, which is another
version of this data set.
}
\author{
Daniel Borcard, <daniel.borcard@umontreal.ca> and Pierre Legendre
<pierre.legendre@umontreal.ca>
}
\keyword{mite}
