% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vlm.R
\name{vlm}
\alias{vlm}
\title{Create over time variable plots and summary statistics for variable level monitoring}
\usage{
vlm(dataFl, dateNm, labelFl = NULL, outFl = "otvplots", genCSV = TRUE,
  dataNeedPrep = FALSE, dateGp = NULL, dateGpBp = NULL, weightNm = NULL,
  varNms = NULL, sortVars = NULL, sortFn = NULL, selectCols = NULL,
  dropCols = NULL, dateFt = "\%d\%h\%Y", buildTm = NULL,
  highlightNms = NULL, skewOpt = NULL, kSample = 50000,
  fuzzyLabelFn = NULL, dropConstants = FALSE, kCategories = 9, ...)
}
\arguments{
\item{dataFl}{Either the name of an object that can be converted using
\code{\link[data.table]{as.data.table}} (e.g., a data frame), or a 
character string containing the name of dataset that can be loaded using 
\code{\link[data.table]{fread}} (e.g., a csv file). If the dataset is not in 
your working directory then \code{dataFl} must include (relative or 
absolute) path to file.}

\item{dateNm}{Name of column containing the date variable.}

\item{labelFl}{Either the path of a dataset (a csv file) containing
labels, an R object convertible to \code{data.table} (e.g., data frame) or 
\code{NULL}. If \code{NULL}, no labels will be used. The label dataset must 
contain at least 2 columns: \code{varCol} (variable names) and 
\code{labelCol} (variable labels).}

\item{outFl}{Name of the output file, with no extension names (e.g., "bank"). 
A pdf file of plots ("bank.pdf"), and two csv files of summary statistics
("bank_categorical_summary.csv" and "bank_numerical_summary.csv") will be
saved to your working directory, unless a path is included in \code{outFl}
(e.g. "../plots/bank").}

\item{genCSV}{Logical, whether to generate the two csv files of summary
statistics for numerical and categorical variables.}

\item{dataNeedPrep}{Logical, indicates if data should be run through the 
\code{\link{PrepData}} function. This should be set to \code{TRUE} unless 
the \code{\link{PrepData}} function has been applied to the input data 
\code{dataFl}.}

\item{dateGp}{Name of the variable that the time series plots should be 
grouped by. Options are \code{NULL}, \code{"weeks"}, \code{"months"}, 
\code{"quarters"}, \code{"years"}. See \code{\link[data.table]{IDate}} for
details. If \code{NULL}, then \code{dateNm} will be used as \code{dateGp}.}

\item{dateGpBp}{Name of variable the boxplots should be grouped by. Same
options as \code{dateGp}. If \code{NULL}, then \code{dateGp} will be used.}

\item{weightNm}{Name of the variable containing row weights, or \code{NULL} for 
no weights (all rows receiving weight 1).}

\item{varNms}{Either \code{NULL} or a vector of names or indices of variables
to be plotted. If \code{NULL}, will default to all columns which are not 
\code{dateNm} or \code{weightNm}. Can also be a vector of indices of the 
column names, after \code{dropCols} or \code{selectCols} have been applied,
if applicable, and not including \code{dateGp}, \code{dateGpBp} 
(which will be added to the \code{dataFl} by the function 
\code{\link{PrepData}}).}

\item{sortVars}{Determines which variables to be plotted and their order. 
Either a character vector of variable names to plot variables in the same
order as in the \code{sortVars} argument), or \code{NULL} to keep the 
original ordering, with numerical variables will being plotted before 
categorical and binary ones. \code{sortVars} should be \code{NULL} when the
\code{sortFn} argument is used.}

\item{sortFn}{A sorting function which returns \code{sortVars} as an output. 
The function may take the following variables as input: \code{dataFl}, 
\code{dateNm}, \code{buildTm}, \code{weightNm}, \code{kSample}. Currently, 
the only build-in sorting function is \code{\link{OrderByR2}}, which sorts
numerical variables in the order of strength of linear association with date,
and adds categorical (and binary) variables sorted in alphabetical order
after the numerical ones.}

\item{selectCols}{Either \code{NULL}, or a vector of names or indices of 
variables to read into memory -- must include \code{dateNm}, 
\code{weightNm} (if not \code{NULL}) and all variables to be plotted. If
both \code{selectCols} and \code{dropCols} are \code{NULL}, then all
variables will be read in.}

\item{dropCols}{Either \code{NULL}, or a vector of variables names or indices
of variables not to read into memory. If both \code{selectCols} and 
\code{dropCols} are \code{NULL}, then all variables will be read in.}

\item{dateFt}{\code{\link{strptime}} format of date variable. The default is SAS
format \code{"\%d\%h\%Y"}. But input data with R date format 
\code{"\%Y-\%m-\%d"} will also be detected. Both of two formats can be
parsed automatically.}

\item{buildTm}{Vector identify time period for ranking/anomaly detection
(most likely model build period). Allows for a subset of plotting time
period to be used for anomaly detection.
\itemize{
     \item Must be a vector of dates and must be inclusive i.e. buildTm[1]
       <= date <= buildTm[2] will define the time period.
     \item Must be either \code{NULL}, a vector of length 2, or a vector of 
       length 3. 
     \item If \code{NULL}, the entire dataset will be used for 
       ranking/anomaly detection. 
     \item If a vector of length 2, the format of the dates must be
       a character vector in default R date format (e.g. "2017-01-30"). 
     \item If a vector of length 3, the first two columns must contain dates 
       in any strptime format, while the 3rd column contains the strptime 
       format (see \code{\link{strptime}}). 
     \item The following are equivalent ways of selecting
       all of 2014:
     \itemize{
       \item \code{c("2014-01-01","2014-12-31")}
       \item \code{c("01JAN2014","31DEC2014", "\%d\%h\%Y")}
     }
}}

\item{highlightNms}{Either \code{NULL} or a character vector of variables to
recieve red label. Currently \code{NULL} means all variables will get a 
black legend. Ignored this argument if \code{labelFl == NULL}.}

\item{skewOpt}{Either a numeric constant or \code{NULL}. Default is 
\code{NULL} (no transformation). If numeric, say 5, then all box plots of
a variable whose skewness exceeds 5 will be on a log10 scale if possible.
Negative input of \code{skewOpt} will be converted to 3.}

\item{kSample}{Either \code{NULL} or a positive integer. If an integer, 
indicates the sample size for both drawing boxplots and ordering numerical
graphs by \eqn{R^2}. When the data is large, setting \code{kSample} to a 
reasonable value (default is 50K) dramatically improves processing speed. 
Therefore, for larger datasets (e.g. > 10 percent system memory), this
parameter should not be set to \code{NULL}, or boxplots may take a very
long time to render. This setting has no impact on the accuracy of time 
series plots on quantiles, mean, SD, and missing and zero rates.}

\item{fuzzyLabelFn}{Either \code{NULL} or a function of 2 parameters: A label
file in the format of an output by \code{\link{PrepLabels}} and a string
giving a variable name. The function should return the label corresponding
to the variable given by the second parameter. This function should 
describe how fuzzy matching should be performed to find labels (see example
below). If \code{NULL}, only exact matches will be retuned.}

\item{dropConstants}{Logical, indicates whether or not constant (all
duplicated or NA) variables should be dropped from \code{dataFl} prior to
plotting.}

\item{kCategories}{If a categorical variable has more than \code{kCategories},
trace plots of only the \code{kCategories} most prevalent categories are
plotted.}

\item{...}{Additional parameters to be passed to 
\code{\link[data.table]{fread}}.}
}
\description{
Sorts variables according to either user input or correlation with time 
(among numerical variables only), and create output files including:
\itemize{
 \item A PDF file of plots saved as \code{outFl}.pdf, with each indivual page 
 on one variable. Variables are plotted in the order indicated in the argument
 \code{sortVars} or \code{sortFn}. 
 For each numerical variable, the output plots include 
 \itemize{
   \item side-by-side boxplots grouped by \code{dateGpBp} (left), 
   \item a trace plot of p1, p50, and p99 percentiles, grouped by \code{dateGp}
     (top right), 
   \item a trace plot of mean and +-1 SD control limits, grouped by 
     \code{dateGp}(middle right), and 
   \item a trace plot of missing and zerorates, grouped by \code{dateGp} 
     (bottom right).
  }
  For each categorical variable (including a numerical variable with no more 
  than 2 unique levels not including NA), the output plots include 
  \itemize{
    \item a frequency bar plot (left), and 
    \item a grid of trace plots on categories' proportions over time (right). 
      If the variable contains more than \code{kCategories} number of 
      categories, trace plots of only the largest \code{kCategories} will be 
      plotted. If the variable contains only two categories, then only the 
      trace plot of the less prevalent cateogy will be plotted.
  }
  \item CSV file(s) on summary statistics of variable, both globally and over
  time aggregated by \code{dateGp}. The order of variables in the CSV files
  are the same as in the PDF file. 
  \itemize{
    \item For numerical varaibles, number of observations (counts), p1, p25, 
    p50, p75, and p99 qunatiles, mean, SD, missing and zerorates are saved
    as \code{outFl}_numerical_summary.csv.
    \item For categorical varaibles, number of observations (counts) and 
    categories' proportions are saved as \code{outFl}_categorical_summary.csv. 
    Each row is a category of a categorical (or binary) variable.
    The row whose \code{category == 'NA'} corresponds to missing. Categories
    among the same variable are ordered by global prevalence in a descending 
    order.
  }
}
}
\details{
If the argument \code{dataNeedPrep} is set to \code{FALSE}, then 
\itemize{
\item \code{dataFl} must be a \code{data.table} containing variables 
  \code{weightNm}, \code{dateNm}, \code{dateGp}, and \code{dateGpBp}, and 
  names of these variables must be the same as the corresponding arguments
  of the \code{\link{vlm}} function.
\item the arguments \code{selectCols}, \code{dropCols}, \code{dateFt}, 
  \code{dropConstants} will be ignored by the \code{\link{vlm}} function.
\item When analyzing a dataset for the first time, it is recommended to first
  run the \code{\link{PrepData}} function on it, and then apply the 
  \code{\link{vlm}} function with the argument \code{dataNeedPrep = FALSE}.
  Please see the examples for details. 
}
}
\section{License}{
 Copyright 2017 Capital One Services, LLC Licensed under the
Apache License, Version 2.0 (the "License"); you may not use this file 
except in compliance with the License. You may obtain a copy of the License
at http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable
law or agreed to in writing, software distributed under the License is
distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied. See the License for the specific language
governing permissions and limitations under the License.
}

\examples{
## Load the data and its label
data(bankData)
data(bankLabels)

## The PrepData function should only need to be run once on a dataset, 
## after that vlm can be run with the argument dataNeedPrep = FALSE
bankData <- PrepData(bankData, dateNm = "date", dateGp = "months", 
                    dateGpBp = "quarters")
bankLabels <- PrepLabels(bankLabels)

\dontrun{ 
vlm(dataFl = bankData, dateNm = "date", labelFl = bankLabels, 
    sortFn = "OrderByR2", dateGp = "months", dateGpBp = "quarters", 
    outFl = "bank")
    
## If csv files of summary statistics are not need, set genCSV = FALSE
vlm(dataFl = bankData, dateNm = "date", labelFl = bankLabels, genCSV = FALSE,
    sortFn = "OrderByR2", dateGp = "months", dateGpBp = "quarters", 
    outFl = "bank")
    
## If weights are provided, they will be used in all statistical calculations
bankData[, weight := rnorm(.N, 1, .1)]
vlm(dataFl = bankData, dateNm = "date", labelFl = bankLabels,
    dateGp = "months", dateGpBp = "quarters", weightNm = "weight", 
    outFl = "bank")

## Customize plotting order by passing a vector of variable names to 
## sortVars, but the "date" column must be excluded from sortVars
sortVars <- sort(bankLabels[varCol!="date", varCol])
vlm(dataFl = bankData, dateNm = "date", labelFl = bankLabels, 
    dateGp = "months", dateGpBp = "quarters", outFl = "bank", 
    sortVars = sortVars)
            
## Create plots for a specific variable using the varNms parameter
vlm(dataFl = bankData, dateNm = "date", labelFl = bankLabels, 
    dateGp = "months", dateGpBp = "quarters", outFl = "bank", 
    varNms = "age", sortVars = NULL)
}
}
\seealso{
This function depends on:
         \code{\link{PrintPlots}},
         \code{\link{OrderByR2}},
         \code{\link{PrepData}},
         \code{\link{PrepLabels}}.
}
