\name{matched}
\alias{matched}
\alias{unmatched}
\alias{matchfailed}
\title{Identification of units placed into matched sets}
\description{
  Given  a bipartite matching (object of class \code{optmatch}),
  create a logical vector of the same length indicating which
  units were and were not placed into matched sets.
}

\usage{
matched(matchobject)
unmatched(matchobject)
matchfailed(matchobject)
}
\arguments{
  \item{matchobject}{ Vector of class \code{optmatch} (especially
  as generated by a call to \code{fullmatch}). }
}
\details{
  \code{matched} and \code{unmatched} indicate which elements of
  \code{matchobject} do and do not belong to matched sets, as
  indicated by their character representations in
  \code{matchobject}.  

  When \code{fullmatch} has been presented with an inconsistent 
  combination of constraints and discrepancies between potential
  matches, so that there exists no matching (i) with finite total
  discrepancy within matched sets that (ii) respects the given
  constraints, then the matching problem is said to be
  infeasible.  \code{TRUE}s in the output of \code{matchfailed} 
  indicate that this has occurred.  
  
}
\value{
  A logical vector (without names).
}
\author{Ben Hansen}
\note{

  To understand the output of
  \code{matchfailed} element-wise, note that \code{fullmatch}
  handles a matching problem in three steps.  First, if
  \code{fullmatch} has been directed to match within subclasses,
  then it divides its matching problem into a subproblem for each
  subclass.  Second,  \code{fullmatch} removes
  from each subproblem those individual units that lack
  permissible potential matches (i.e. potential matches from
  which they are separated by a finite discrepancy).
  Such ``isolated'' units are flagged in such a way as to be
  indicated by \code{unmatched}, but not by \code{matchfailed}.
  Third, \code{fullmatch} presents each subproblem, with isolated
  elements removed, to an optimal matching routine.  If such a
  reduced subproblem is found at this stage to be infeasible,
  then each unit contributing to it is so flagged as to be
  indicated by \code{matchfailed}. }

\seealso{ \code{\link{fullmatch}} }
\examples{
plantdist <- matrix(nrow=7, ncol=19,byrow=TRUE,data=c(
28, 0, 3,22,14,30,17,28,26,28,20,22,23,26,21,18,34,40,28,
24, 3, 0,22,10,27,14,26,24,24,16,19,20,23,18,16,31,37,25,
10,18,14,18, 4,12, 6,11, 9,10,14,12, 6,14,22,10,16,22,28,
 7,28,24, 8,14, 2,10, 6,12, 0,24,22, 4,24,32,20,18,16,38,
17,20,16,32,18,26,20,18,12,24, 0, 2,20, 6, 8, 4,14,20,14,
20,31,28,35,20,29,22,20,14,26,12, 9,22, 5,15,12, 9,11,12,
14,32,29,30,18,24,17,16,10,22,12,10,17, 6,16,14, 4, 8,17),
dimnames=list(c("A","B","C","D","E","F","G"),
c("H","I","J","K","L","M","N","O","P","Q","R",
"S","T","U","V","W","X","Y","Z")))

mxpl.fm0 <- fullmatch(plantdist) # A feasible matching problem
c(sum(matched(mxpl.fm0)), sum(unmatched(mxpl.fm0)))
sum(matchfailed(mxpl.fm0)) 
mxpl.fm1 <- fullmatch(plantdist, # An infeasible problem
                   max.controls=3, min.controls=3)
c(sum(matched(mxpl.fm1)), sum(unmatched(mxpl.fm1)))
sum(matchfailed(mxpl.fm1))

mxpl.si <- factor(c('a', 'a', 'c', rep('d',4), 'b', 'c', 'c', rep('d', 16)))
names(mxpl.si) <- LETTERS[1:26]
# Subclass a contains two treated units but no controls;
# subclass b contains only a control unit;
# subclass c contains one treated and two control units;
# subclass d contains the remaining twenty units.

mcl <- c(Inf, Inf, 1, Inf)
names(mcl) <- c('a', 'b', 'c', 'd')

mxpl.fm2 <- fullmatch(plantdist,        # contains four subproblems, 
                   subclass.indices=    # three of which are infeasible
                   mxpl.si,
                   max.controls=mcl)
sum(matched(mxpl.fm2))

table(unmatched(mxpl.fm2), matchfailed(mxpl.fm2))

mxpl.fm2[matchfailed(mxpl.fm2)]

mxpl.fm2[unmatched(mxpl.fm2) &   # isolated units return as
         !matchfailed(mxpl.fm2)] # unmatched but not matchfailed

  }

\keyword{manip}
%\keyword{ ~kwd2 } __ONLY ONE__ keyword per line
