###################################################################
## TODO
## - think about save.spline.fit option; perhaps this can be done
##   by having all the fittig in an extra function which doesn't
##   return the spline within do_aggr but does return it elsewise
## - why is "shortcut" in the METHODS constant?
## - I am not sure that the AUC function in fast_estimate works correctly!
##   The indexing seems to be at the wrong point.
## - %*% is a correct operator (and doesn't need to be written as % * % !!!)
## - in fast_estimate, one should swap x and y:
##     y <- x[, time.pos]                 ## predictor (x-axis)
##     x <- x[, -time.pos, drop = FALSE]  ## outcome (y-axis)
##
########################################################
## TODO2:
##
## 2) Implement functions to
## - extract e.g. the (beta) coefficients of the fitted curve,
##
## 3) Define further function-arguments, which should be easily
## specified by the user
##
## 4) Diskuss how and if we want to store the models. Perhaps one
## could use a global option with default
##    options("save.models" = FALSE)
## If users now define (perhaps using a helper function)
##     options("save.models" = TRUE)
## the models could, for example, be saved as .Rda objects
########################################################




################################################################################

#' @title Fit splines
#' @description Fit various splines (smoothing splines and P-splines from
#'   \pkg{mgcv} and smoothing splines via \code{smooth.spline}) to opm data.
#'
#' @param y Character scalar. Outcome. Usually the name of a well.
#' @param x Character scalar. Predictor. Usually \code{"Hour"}.
#' @param data A \code{data.frame} containing the variables specified in
#'   \code{x} and \code{y}.
#' @param options List of options. Generated by
#'   \code{\link{set_spline_options}}.
#' @param weights vector of weights used in the fitting process.
#' @return Spline fit. A model of class  \code{"opm_model"}.
#'
#' @seealso \code{\link{gam}} from \pkg{mgcv} and \code{\link{smooth.spline}}
#' @keywords internal
#~ @keywords smooth
#' @author Benjamin Hofner
fit_spline <- function (y, x = "Hour", data, options = set_spline_options(),
    weights = NULL) {

  call <- match.call()
  type <- options$type
  method <- options$est.method
  knots <- options$knots
  class <- options$class
  if (is.null(weights))
    weights <- rep(1, nrow(data))

  ## compute number of knots adaptive to the number of unique observations
  ## (equal to behavior of smooth.spline(..., nknots = NULL) )
  if (is.null(knots))
    knots <- stats:::n.knots(length(unique(data[, x])))
  ## PERHAPS USE OTHER METHOD TO GET NUMBER OF KNOTS. E.G. BASED ON ROUGHNESS
  ## OF THE DATA

  ## set up model formulae
  if (type == "p.spline" || type == "tp.spline") {
      if (is.null(options$s.par)) {
          fm <- paste(y, " ~ s(", x, ", bs = '", class, "', k = ", knots, ")",
                      sep = "")
      } else {
          fm <- paste(y, " ~ s(", x, ", bs = '", class, "', k = ", knots, ",",
                      options$s.par, ")", sep = "")
      }
      fm <- as.formula(fm)
  } else {
      ## REMOVED ", ..." after knots.
      ## Reintroduce possibility to "..." via options
      fm <- paste("list(x = data$", x, ", y = data$", y,
                  ", penalty = ", options$gamma, ", nknots = ", knots,
                  ", w = weights)",
                  sep = "")
  }
  if (type == "p.spline" || type == "tp.spline") {
      ## REMOVED ... from gam.
      ## Reintroduce possibility to "..." via options
      mod <- gam(fm, data = data, gamma = options$gamma, method = method,
        weights = weights)
      mod$call <- call
  } else {
      mod <- do.call("smooth.spline", eval(parse(text = fm)))
      mod$call <- call
      mod$names <- c(x, y)
  }
  class(mod) <- c("opm_model", paste(class, "_model", sep = ""), class(mod))
  return(mod)
}

################################################################################

#' @title Spline options
#' @description Function to set up spline options which can be passed to
#'   \code{\link{do_aggr}}. %and \code{\link{fit_spline}}.
#'
#' @param type Character scalar. Specifies the spline type which should be
#'   fitted. This can be either thin plate splines (\dQuote{tp.spline}),
#'   penalized B-splines (i.e, P-splines \dQuote{p.spline}) or smoothing
#'   splines (\dQuote{smooth.spline}).
#' @param knots Integer scalar. Determines the number of knots. Per default, the
#'   number of knots is chosen adaptively to the number of unique observations.
#' @param gamma Integer scalar. Specifies a constant multiplier to inflate the
#'   degrees of freedom in the \code{"GCV"} \code{method} to increase
#'   penalization of models that are too close to the data and thus not
#'   smooth enough.
#' @param est.method Character scalar. The smoothing parameter estimation
#'   method. Currently, only \code{"REML"} and \code{"GCV"} are supported. This
#'   argument is ignored for \code{type = "smooth.spline"}. For details see
#'   \code{\link[mgcv]{gam}} (see package \pkg{mgcv}).
#' @param s.par list. Further arguments to be passed to the smoother
#'   \code{\link[mgcv]{s}} (see package \pkg{mgcv}). Note that the \pkg{mgcv}
#'   options \code{k} and \code{bs} are specified using \code{type} and
#'   \code{knots} in \pkg{opm}.
#' @param save.models Should the models be saved (on the disk) for further
#'   inspections and plotting?
#' @param filename Filename of the models. Per default a name is auto-generated
#'   based on date and time. The file is always generated in the current working
#'   directory.
#' @param ... Additional arguments to be passed to \code{\link[mgcv]{gam}} or
#'   \code{\link{smooth.spline}}.
#' @return List of options.
#'
#' @keywords misc
#' @author Benjamin Hofner
#' @export
set_spline_options <- function(type = c("tp.spline",
    "p.spline", "smooth.spline"),
    knots = NULL, gamma = 1, est.method = c("REML", "GCV"), s.par = NULL,
    save.models = FALSE, filename = NULL, ...) {

  if (!missing(...))
    warning(sQuote("..."), " currently not passed to fitting functions")
  type <- match.arg(type)
  class <- ifelse(type == "tp.spline", "tp",
    ifelse(type == "p.spline", "psp", "smooth.spline"))

  method <- match.arg(est.method)
  if (type == "smoothing-splines" && !is.null(s.par))
    warning(sQuote("s.par"), " ignored if ",
      sQuote('type = "smoothing-splines"'))
  if (!is.null(filename) && !save.models) {
    save.models <- TRUE
    warning(sQuote("filename"), " specified, ", sQuote("save.models"),
      " set to TRUE")
  }
  list(type = type, knots = knots, gamma = gamma, est.method = method,
    s.par = s.par, save.models = save.models, filename = filename,
    class = class, ...)
}

################################################################################
## special predict function for models from smooth.spline

#' @title Predictions for smoothing spline models
#'
#' @description Make prediction for estimated splines.
#'
#' @param object An object of class \code{smooth.spline_model}.
#' @param newdata data.frame containing new data for predictions. This is
#'   optional.
#' @param ... additional options. Currently not used.
#' @author Benjamin Hofner
#'
## CURRENTLY INTERNAL
#' @keywords internal
#~ @keywords methods
#'
#' @method predict smooth.spline_model
#' @S3method predict smooth.spline_model
predict.smooth.spline_model <- function(object, newdata = NULL, ...) {
    if (is.null(newdata)) {
        ## get data from fitted model
        newdata <- get_data(object)
    }
    newX <- newdata[, object$names[1]]
    stats:::predict.smooth.spline(object, x = newX, deriv = 0)$y
}

################################################################################
## generic function used to plot spline fit

#' @title Plot spline fits
#' @description Plot splines derived from \code{\link{do_aggr}} or
#'   \code{\link{fit_spline}}.
#'
#' @param x Spline fit.
#' @param plot.data Logical. Should the data be added to the plot?
#' @param plot.spline Logical. Should the spline be plotted?
#' @param col Color of observed data. Per default a semi-transparent grey
#'   value is used. For details see \code{\link{par}} (Section Color
#'   Specification).
#' @param pch Plotting character used for observed data points.
#'   See \code{\link{points}} for details.
#' @param col.spline Color of fitted spline specified for example by a character
#'   string. For details see \code{\link{par}} (Section Color Specification).
#' @param lty.spline Line type of fitted spline.
#'   For details see \code{\link{par}}.
#' @param lwd.spline Line width of fitted spline.
#'   For details see \code{\link{par}}.
#' @param ... Further arguments to be passed to \code{\link{plot}} and
#'   \code{\link{lines}}.
#'
#' @author Benjamin Hofner
#'
## CURRENTLY NOT EXPORTED
#' @keywords internal
#~ @keywords hplot methods
#'
#' @method plot opm_model
#' @S3method plot opm_model
plot.opm_model <- function(x, plot.data = TRUE, plot.spline = TRUE,
    col = rgb(0, 0, 0, 0.3), pch = 20, col.spline = "red",
    lty.spline = "solid", lwd.spline = 1, ...) {

    if (!plot.data && ! plot.spline)
        stop("Nothing to be plotted")
    data <- get_data(x)
    plot_helper(data, col = col, pch = pch, plot.data = plot.data, ...)
    if (plot.spline)
        lines(x, col = col.spline, lty = lty.spline, lwd = lwd.spline, ...)
}

################################################################################
## generic function used to add spline fits to existing plots

#' @title Add spline fits to plot
#' @description Add a fitted spline function to an existing plot.
#'
#' @param x Spline fit.
#' @param col Color of fitted spline specified for example by a character
#'   string. For details see \code{\link{par}} (Section Color Specification).
#' @param lty Line type of fitted spline. For details see \code{\link{par}}.
#' @param lwd Line width of fitted spline. For details see \code{\link{par}}.
#' @param ... Further arguments to be passed to \code{\link{lines}}
#' @author Benjamin Hofner
#'
## CURRENTLY NOT EXPORTED
#' @keywords internal
#~ @keywords aplot methods
#'
#' @method lines opm_model
#' @S3method lines opm_model
lines.opm_model <- function(x, col = "red", lty = "solid", lwd = 1, ...) {
    pred <- predict(x)
    x <- get_data(x)[, 1]
    ix <- order(x)
    lines(x[ix], pred[ix], col = col, lty = lty, lwd = lwd, ...)
}


################################################################################

#' @title Compute AUC
#' @description Compute AUC (area under the curve) using the trapezoidal rule.
#'
#' @param x Numeric vector.
#' @param y Numeric vector.
#' @param index possible index vector if only a subset should be used.
#' @author Benjamin Hofner
#' @keywords internal
#~ @keywords utilities
AUC <- function(x, y, index = NULL) {
  if (!is.null(index)) {
      x <- x[index]
      y <- y[index]
  }
  n.obs <- length(x)
  sum(diff(x) * (0.5 * (y[-1L] + y[-n.obs])))
}

################################################################################
## make P-splines that allow predictions at or outside of the boundary knots
## modified version of mgcv's ps smooth.construct.ps.smooth.spec

#' @title Constructor for P-splines
#' @description A modified version of the P-spline constructor
#'   \code{\link{smooth.construct.ps.smooth.spec}} from package
#'   \pkg{mgcv} that allows to make at or outside of the boundary knots.
#'
#' @param object a smooth specification object, usually generated by the term
#'   \code{s(x, bs = "psp", ...)}.
#' @param data a list containing just the data required by this term
#'   with names corresponding to \code{object$term}.
#' @param knots a list containing any knots supplied for basis setup in
#'   same order and with same names as \code{data}.
#' @details This function is primarily for internal use. For details see
#'   \code{\link{smooth.construct}} and
#'   \code{\link{smooth.construct.ps.smooth.spec}}.
#' @return An object of class \code{psp.smooth}.
#' @author Benjamin Hofner bases on work by Simon N. Wood
#'
#' @keywords internal
#'
#' @method smooth.construct psp.smooth.spec
#' @S3method smooth.construct psp.smooth.spec
smooth.construct.psp.smooth.spec <- function (object, data, knots) {
    if (length(object$p.order) == 1)
        m <- rep(object$p.order, 2)
    else m <- object$p.order
    m[is.na(m)] <- 2
    object$p.order <- m
    if (object$bs.dim < 0)
        object$bs.dim <- max(10, m[1] + 1)
    nk <- object$bs.dim - m[1]
    if (nk <= 0)
        stop("basis dimension too small for b-spline order")
    if (length(object$term) != 1)
        stop("Basis only handles 1D smooths")
    x <- data[[object$term]]
    k <- knots[[object$term]]
    if (is.null(k)) {
        xl <- min(x)
        xu <- max(x)
    }
    else if (length(k) == 2) {
        xl <- min(k)
        xu <- max(k)
        if (xl > min(x) || xu < max(x))
            stop("knot range does not include data")
    }
    if (is.null(k) || length(k) == 2) {
        xr <- xu - xl
        xl <- xl - xr * 0.001
        xu <- xu + xr * 0.001
        dx <- (xu - xl) / (nk - 1)
        k <- seq(xl - dx * (m[1] + 1), xu + dx * (m[1] + 1),
            length = nk + 2 * m[1] + 2)
    }
    else {
        if (length(k) != nk + 2 * m[1] + 2)
            stop(paste("there should be ", nk + 2 * m[1] + 2,
                " supplied knots"))
    }
    object$X <- spline.des(k, x, m[1] + 2, x * 0, outer.ok = TRUE)$design
    if (!is.null(k)) {
        if (sum(colSums(object$X) == 0) > 0)
            warning("knot range is so wide that there is *no* information",
                    " about some basis coefficients")
    }
    S <- diag(object$bs.dim)
    if (m[2])
        for (i in 1:m[2]) S <- diff(S)
    object$S <- list(t(S) %*% S)
    object$S[[1]] <- (object$S[[1]] + t(object$S[[1]])) / 2
    object$rank <- object$bs.dim - m[2]
    object$null.space.dim <- m[2]
    object$knots <- k
    object$m <- m
    class(object) <- "psp.smooth"
    object
}

################################################################################
## function for psp smooths as defined above
## needed to make predictions at or outside of the boundarie knots

#' @title Prediction method for P-splines
#' @description Prediction method for P-splines fitted via
#'   \code{gam(y ~ s(x, bs = "psp", ...))}. Used internally in
#'   \code{\link{do_aggr}} with \code{method = "spline.fit"} and
#'   \code{set_spline_options(type = "p.spline")}.
#'
#' @param object An object of class \code{psp.smooth}.
#' @param data A \code{data.frame} for which the smooth term is to be evaluated.
#'
#' @return Matrix.
#'
#' @author Benjamin Hofner
#' @keywords internal
#'
#' @method Predict.matrix psp.smooth
#' @S3method Predict.matrix psp.smooth
#'
Predict.matrix.psp.smooth <- function (object, data) {
    X <- spline.des(object$knots, data[[object$term]], object$m[1] + 2,
                    outer.ok = TRUE)$design
    X
}

################################################################################

#' @title Add parameter estimates
#' @description Add parameter estimates for an estimated spline model to the
#' current plot.
#'
#' @param model Model fit.
#' @param add.deriv Should the derivative be plotted?
#' @param col Color for parameter estimates.
#' @param deriv.col Color for first derivative.
#' @param lty Line type.
#' @param ... Additional graphical parameters that are passed to
#'   \code{\link{lines}}, \code{\link{points}} and \code{\link{abline}}.
#' @return NULL
#' @keywords internal
#~ @keywords aplot
#'
#' @author Benjamin Hofner
add_parameters <- function(model, add.deriv = FALSE, col = "red",
  deriv.col = "grey", lty = "dashed", ...) {

    x <- extract_curve_params.opm_model(model, all = TRUE)
    data <- get_data(model)[, 1]
    if (add.deriv) {
        lines(data[-1], x$deriv, col = deriv.col, lty = lty, ...)
        abline(h = 0, lty = lty, col = deriv.col)
    }
    abline(a = x$intercept, b = x$mu, col = col, lty = lty, ...)
    points(x$lambda, 0, col = col, ...)
    abline(h = x$A, col = col, lty = lty, ...)
}


################################################################################

## (internal) function used to generate the plot area and draw the raw data (if
## plot.data = TRUE)

#' @title Helper function for plotting
#' @description Helper function for plotting internally used by
#'   \code{\link{plot.opm_model}}. Sets up a device and plots raw data if asked
#'   to.
#' @param data raw data.
#' @param plot.data Logical. Should the raw data be plotted?
#' @param col Color for plotting.
#' @param pch Point symbols to be plotted.
#' @param ... Further arguments to be passed to \code{\link{plot}} and
#'   \code{\link{points}}.
#' @return NULL
#' @author Benjamin Hofner
#' @keywords internal
plot_helper <- function(data, plot.data = TRUE, col = rgb(0, 0, 0, 0.3),
                        pch = 20, ...) {
    ## generate empty plot of appropriate size
    plot(data[, 1], data[, 2],
      ylab = names(data)[2],
      xlab = names(data)[1],
      type = "n", ...)
    if (plot.data) {
        points(data[, 1], data[, 2], col = col, pch = pch, ...)
    }
}

################################################################################

## (internal) generic needed to extract the original data from various spline
## estimates

#' @title Extract data
#' @description Extract data from various spline estimates
#' @param x Model.
#' @return data.frame
#' @author Benjamin Hofner
#' @keywords internal
#'
get_data <- function(x)
    UseMethod("get_data")

## (internal) function for psp and tp models (derived via mgcv)

#' @rdname get_data
#' @method get_data psp_model
#' @S3method get_data psp_model
#' @method get_data tp_model
#' @S3method get_data tp_model
get_data.psp_model <- get_data.tp_model <- function(x) {
    data <- x$model
    data <- data[, c(2, 1)]
    return(data)
}

## (internal) function for ss models (derived via smooth.spline)

#' @rdname get_data
#' @method get_data smooth.spline_model
#' @S3method get_data smooth.spline_model
get_data.smooth.spline_model <- function(x) {
    data <- as.data.frame(x$data)
    names(data)[1:2] <- x$names
    return(data)
}
