% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bin_table.R
\name{bin_table}
\alias{bin_table}
\title{Bin table of environmental conditions in M and for occurrences}
\usage{
bin_table(Ms, occurrences, species, longitude, latitude, variable,
  percentage_out = 5, bin_size = 10)
}
\arguments{
\item{Ms}{a list of SpatialPolygons* objects representing the accessible area
(M) for all species to be analyzed. The order of species represented by each
object here must coincide with the one in \code{occurrences}. See details.}

\item{occurrences}{a list of data.frames of occurrence records for all species.
The order of species represented by each data.frame must coincide with the one
in \code{Ms}. See details.}

\item{species}{(character) name of the column in occurrence data.frames that
contains the name of the species.}

\item{longitude}{(character) name of the column in occurrence files containing
values of longitude.}

\item{latitude}{(character) name of the column in occurrence files containing
values of latitude.}

\item{variable}{a RasterLayer of an environmental variable of interest.
See details.}

\item{percentage_out}{(numeric) percentage of extreme environmental data in M
to be excluded in bin creation for further analyses. See details. Default = 5.}

\item{bin_size}{(numeric) size of bins. Range of environmental values to
be considered when creating each character in bin tables. See details.
Default = 10.}
}
\value{
A list containing a table of characters to represent ecological niches of the
species of interest.

Potential values for characters are:
\itemize{
\item "1" = the species is present in those environmental conditions.
\item "0" = the species is not present in those environmental conditions. This is,
those environmental conditions inside the accessible area (M) are more extreme
than the ones used for the species.
\item "?" = there is no certainty about the species presence in those environmental
conditions. This happens in environmental combinations more extreme than the
ones found in the accessible area (M), when environmental conditions in
species records are as extreme as the most extreme ones in M.
}
}
\description{
bin_table helps in creating a bin table of environmental
conditions in accessible areas (M) and for species occurrence records
(i.e., table of characters).
}
\details{
Coordinates in \code{occurrences}, SpatialPolygons* objects in \code{Ms}, and
RasterLayer in \code{variable} must coincide in the geographic projection in
which they are represented. WGS84 with no planar projection is recommended.

Accessible area (M) is understood as the geographic area that has been
accessible for a species for relevant periods of time. Defining M is usually
a hard task, but also a very important one, because it allows identifying
uncertainties about the ability of a species to maintain populations in
certain environmental conditions. For further details on this topic, see
Barve et al. (2011) in \url{https://doi.org/10.1016/j.ecolmodel.2011.02.011}.

The percentage to be defined in \code{percentage_out} excludes a percentage
of extreme environmental values to prevent from considering extremely rare
environmental values in the accessible area for the species (M). Being too
rare, these values may have never been explored by the species; therefore,
including them in the process of preparation of the table of characters
(bin table) is risky.

The argument \code{bin_size} helps to create characters that represent not
only one value of an environmental variable, but a range of environmental
conditions. For instance, if a variable of precipitation in mm is used, a
value of 10 for \code{bin_size} indicates that each character will represent
a class that correspond to 10 continuous values of precipitation (e.g., from
100 to 110 mm).
}
\examples{
# getting a variable at coarse resolution
\donttest{
tmpd <- file.path(tempdir(), "bios") # temporal directory
dir.create(tmpd)
temp <- raster::getData("worldclim", var = "bio", res = 10, path = tmpd)[[1]]

# example data
data("m_list", package = "nichevol")
data("occ_list", package = "nichevol")

# preparing bins
char_table <- bin_table(Ms = m_list, occurrences = occ_list, species = "species",
                        longitude = "x", latitude = "y", variable = temp,
                        percentage_out = 5, bin_size = 10)
}
}
