% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/indirect.relations.R
\name{indirect_relations}
\alias{indirect_relations}
\title{Indirect relations in a network}
\usage{
indirect_relations(g, type = "dist_sp", lfparam = NULL, dwparam = NULL,
  netflowmode = "", rspxparam = NULL, FUN = identity, ...)
}
\arguments{
\item{g}{igraph object. The network for which relations should be derived.}

\item{type}{String giving the relation to be calculated. See Details for options.}

\item{lfparam}{Numeric parameter. Only used if type = "dist_lf".}

\item{dwparam}{Numeric parameter. Only used if type = "dist_walk".}

\item{netflowmode}{String, one of raw, frac, or norm. Only used if type = "depend_netflow".}

\item{rspxparam}{Numeric parameter. Only used if type = "depend_rsps" or type = "depend_rspn".}

\item{FUN}{A function that allows the transformation of relations. See Details.}

\item{...}{Additional arguments passed to FUN.}
}
\value{
A matrix containing indirect relations in a network.
}
\description{
Derive indirect relations for a given network.
Observed relations, like presents or absence of a relation, are commonly not the center
of analysis, but are transformed in a new set of indirect relation like shortest path
distances among nodes. These transformations are usually an implicit step when centrality
indices are used. Making this step explicit gives more possibilities, for example
calculating partial centrality rankings with \link{positional_dominance}.
}
\details{
The \code{type} parameter has the following options.

\emph{'adjacency'} returns the adjacency matrix of the network.

\emph{'weights'} returns the weighted adjacency matrix of the network if an edge
attribute 'weight' is present.

\emph{'dist_sp'} returns shortest path distances between all pairs of nodes.

\emph{'depend_sp'} returns dyadic dependencies
\deqn{\delta(u,s) = \sum_{t \in V} \frac{\sigma(s,t|u)}{\sigma(s,t)}}
where \eqn{\sigma(s,t|u)} is the number of shortest paths from s to t that include u and
\eqn{\sigma(s,t)} is the total number of shortest (s,t)-paths. This relation is used
for betweenness-like centrality indices.

\emph{'walks'} returns walk counts between pairs of nodes, usually they are
weighted decreasingly in their lengths or other properties which can be done by adding
a function in \code{FUN}.  See \link{transform_relations} for options.

\emph{'dist_resist'} returns the resistance distance between all pairs of nodes.

\emph{'dist_lf'} returns a logarithmic forest distance \eqn{d_\alpha(s,t)}. The logarithmic forest
distances form a one-parametric family of distances, converging to shortest path distances as \eqn{\alpha -> 0}
and to the resistance distance as \eqn{\alpha -> \infty}. See (Chebotarev, 2011) for more details.
The parameter \code{lfparam} can be used to tune \eqn{\alpha}.

\emph{'dist_walk'} returns the walk distance \eqn{d_\alpha^W(s,t)} between nodes. The walk distances form a one-parametric
family of distances, converging to shortest path distances as \eqn{\alpha -> 0} and to longest
walk distances for \eqn{\alpha -> \infty}. Walk distances contain the logarithmic forest
distances as a special case. See (Chebotarev, 2012) for more details.

\emph{'dist_rwalk'} returns the expected length of a random walk between two nodes. For more
details see (Noh and Rieger, 2004)

\emph{'depend_netflow'} returns dependencies based on network flow (See Freeman et al.,1991).
If \code{netflowmode="raw"}, the function returns
\deqn{\delta(u,s) = \sum_{t \in V} f(s,t,G)-f(s,t,G-v)}
where f(s,t,G) is the maximum flow from s to t in G and f(s,t,G-v) in G without the node v.
For \code{netflowmode="frac"} it returns dependencies in the form, similar to shortest path dependencies:
\deqn{\delta(u,s) = \sum_{t \in V} \frac{f(s,t,G)-f(s,t,G-v)}{f(s,t,G)}}

\emph{'depend_curflow'} returns pairwise dependencies based on current flow. The relation is
based on the same idea as 'depend_sp' and 'depend_netflow'. However, instead of considering
shortest paths or network flow, the current flow (or equivalent: random walks) between nodes
are of interest. See (Newman, 2005) for details.

\emph{'depend_exp'} returns pairwise dependencies based on 'communicability':
\deqn{\delta(u,s)=\sum_{t \in V} \frac{exp(A)_{st}-exp(A+E(u))_{st}}{exp(A)_{st}},}
where E(u) has nonzeros only in row and column u, and
in this row and column has -1 if A has +1. See (Estrada et al., 2009) for additional details.

\emph{'depend_rsps'}. Simple randomized shortest path dependencies.
The simple RSP dependency of a node u with respect to absorbing paths from s to t,
is defined as the expected number of visits through u over all s-t-walks. The
parameter \code{rspxparam} is the "inverse temperature parameter".
If it converges to infinity, only shortest paths are considered and the expected
number of visits to a node on a shortest path approaches the probability of
following that particular path. When the parameter converges to zero, then the
dependencies converge to the expected number of visits to a node over all absorbing
walks with respect to the unbiased random walk probabilities. This means for undirected networks,
that the relations converge to adjacency. See (Kivimäki et al., 2016) for details.

\emph{'depend_rspn'} Net randomized shortest path dependencies.
The parameter \code{rspxparam} is the "inverse temperature parameter". The asymptotic
for the infinity case are the same as for 'depend_rsps'. If the parameter approaches zero, then
it converges to 'depend_curflow'. The net randomized shortest path dependencies
are closely related to the random walk interpretation of current flows.
See (Kivimäki et al., 2016) for technical details.

The function \code{FUN} is used to transform the indirect
relation. See \link{transform_relations} for predefined functions and additional help.
}
\examples{
library(igraph)
g <- graph.empty(n=11,directed = FALSE)
g <- add_edges(g,c(1,11,2,4,3,5,3,11,4,8,5,9,5,11,6,7,6,8,
                   6,10,6,11,7,9,7,10,7,11,8,9,8,10,9,10))

#shortest path distances
D <- indirect_relations(g,type = "dist_sp")

#inverted shortest path distances
D <- indirect_relations(g,type = "dist_sp", FUN = dist_inv)
#shortes path dependencies (used for betweenness)
D <- indirect_relations(g,type = "depend_sp")

#walks attenuated exponentially by their length
W <- indirect_relations(g,type = "walks",FUN = walks_exp)

}
\references{
Chebotarev, P., 2012. The walk distances in graphs. \emph{Discrete Applied Mathematics}, 160(10), pp.1484-1500.

Chebotarev, P., 2011. A class of graph-geodetic distances generalizing the shortest-path and
the resistance distances. \emph{Discrete Applied Mathematics} 159,295-302.

Noh, J.D. and Rieger, H., 2004. Random walks on complex networks. \emph{Physical Review Letters}, 92(11), p.118701.

Freeman, L.C., Borgatti, S.P., and White, D.R., 1991.
Centrality in Valued Graphs: A Measure of Betweenness Based on Network Flow. \emph{Social Networks} 13(2), 141-154.

Newman, M.E., 2005. A measure of betweenness centrality based on random walks. \emph{Social Networks}, 27(1), pp.39-54.

Estrada, E., Higham, D.J., and Hatano, N., 2009.
Communicability betweenness in complex networks. \emph{Physica A} 388,764-774.

Kivimäki, I., Lebichot, B., Saramäki, J., and Saerens, M., 2016.
Two betweenness centrality measures based on Randomized Shortest Paths
\emph{Scientific Reports} 6: 19668
}
\seealso{
\link{aggregate_positions} to build centrality indices, \link{positional_dominance} to derive dominance relations
}
\author{
David Schoch
}
