# nanonext - Utilities ---------------------------------------------------------

#' NNG Library Version
#'
#' Returns the version of 'libnng' used and whether TLS is supported.
#'
#' @return A character vector of length 2.
#'
#' @section TLS Support:
#'
#'     The environment variable 'NANONEXT_TLS' may be set, e.g. by
#'     \code{Sys.setenv(NANONEXT_TLS=1)}, prior to package installation to enable
#'     TLS where the system NNG library has been built with TLS support (using
#'     Mbed TLS). Note: this is not applicable to Windows systems.
#'
#' @examples
#' nng_version()
#'
#' @export
#'
nng_version <- function() .Call(rnng_version)

#' Translate Error Codes
#'
#' Translate integer exit code to human readable form. All package functions
#'     return an integer exit code on error rather than the expected return
#'     value. These are classed 'errorValue' and may be checked by the function
#'     \code{\link{is_error_value}}.
#'
#' @param xc integer exit code to translate.
#'
#' @return A character vector.
#'
#' @section Warnings:
#'
#'     A warning is generated every time an 'errorValue' is returned.
#'
#'     \code{\link{nano_init}} may be used to set the value of option 'warn' and
#'     automatically reverts it upon package unload. The default, applied by
#'     calling \code{nano_init()} with no arguments, is 'immediate', which prints
#'     warnings as they occur.
#'
#'     Further options for warnings may be set manually via \code{\link{options}}:
#'
#'     \itemize{
#'
#'     \item{warning.expression} { - an R code expression to be called if a
#'     warning is
#'     generated, replacing the standard message. If non-null it is called
#'     irrespective of the value of option warn.}
#'
#'     \item{warning.length} { - sets the truncation limit in bytes for error and warning
#'     messages. A non-negative integer, with allowed values 100...8170, default
#'     1000.}
#'
#'     \item{nwarnings} { - the limit for the number of warnings kept when warn = 0,
#'     default 50. This will discard messages if called whilst they are being
#'     collected. If you increase this limit, be aware that the current
#'     implementation pre-allocates the equivalent of a named list for them.}
#'     }
#'
#' @examples
#' nng_error(1L)
#'
#' @export
#'
nng_error <- function(xc) .Call(rnng_strerror, as.integer(xc))

#' Is Nul Byte
#'
#' Is the object a nul byte.
#'
#' @param x an object.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @examples
#' is_nul_byte(as.raw(0L))
#' is_nul_byte(raw(length = 1L))
#' is_nul_byte(writeBin("", con = raw()))
#'
#' is_nul_byte(0L)
#' is_nul_byte(NULL)
#' is_nul_byte(NA)
#'
#' @export
#'
is_nul_byte <- function(x) identical(x, as.raw(0L))

#' Is Error Value
#'
#' Is the object an error value generated by NNG. All returned integer error
#'     codes are classed as 'errorValue' to be distinguishable from integer
#'     message values.
#'
#' @param x an object.
#'
#' @return Logical value TRUE if 'x' is of class 'errorValue', FALSE otherwise.
#'
#' @section Warnings:
#'
#'     A warning is generated every time an 'errorValue' is returned.
#'
#'     \code{\link{nano_init}} may be used to set the value of option 'warn' and
#'     automatically reverts it upon package unload. The default, applied by
#'     calling \code{nano_init()} with no arguments, is 'immediate', which prints
#'     warnings as they occur.
#'
#'     Further options for warnings may be set manually via \code{\link{options}}:
#'
#'     \itemize{
#'
#'     \item{warning.expression} { - an R code expression to be called if a
#'     warning is
#'     generated, replacing the standard message. If non-null it is called
#'     irrespective of the value of option warn.}
#'
#'     \item{warning.length} { - sets the truncation limit in bytes for error and warning
#'     messages. A non-negative integer, with allowed values 100...8170, default
#'     1000.}
#'
#'     \item{nwarnings} { - the limit for the number of warnings kept when warn = 0,
#'     default 50. This will discard messages if called whilst they are being
#'     collected. If you increase this limit, be aware that the current
#'     implementation pre-allocates the equivalent of a named list for them.}
#'     }
#'
#' @examples
#' is_error_value(1L)
#'
#' @export
#'
is_error_value <- function(x) inherits(x, "errorValue")

#' nanonext Initialise
#'
#' Initialise global options - intended to be called immediately after package load.
#'
#' @param warn [default 'immediate'] character string defining how to treat
#'     warnings generated by the package. 'immediate' to print warnings as they
#'     occur, 'deferred' to print warnings when evaluation returns to the top
#'     level, 'error' to upgrade all warnings to errors (stops execution), and
#'     'none' to ignore all warnings.
#'
#' @return Invisibly, the integer \code{code} applied to \code{options(warn = code)}.
#'
#' @section Warnings:
#'
#'     A warning is generated every time an 'errorValue' is returned.
#'
#'     This function sets the global option 'warn' to the appropriate value and
#'     automatically reverts it upon package unload. The default, applied by
#'     calling \code{nano_init()} with no arguments, is 'immediate', which
#'     prints warnings as they occur.
#'
#'     Further options for warnings may be set manually via \code{\link{options}}:
#'
#'     \itemize{
#'
#'     \item{warning.expression} { - an R code expression to be called if a
#'     warning is
#'     generated, replacing the standard message. If non-null it is called
#'     irrespective of the value of option warn.}
#'
#'     \item{warning.length} { - sets the truncation limit in bytes for error and warning
#'     messages. A non-negative integer, with allowed values 100...8170, default
#'     1000.}
#'
#'     \item{nwarnings} { - the limit for the number of warnings kept when warn = 0,
#'     default 50. This will discard messages if called whilst they are being
#'     collected. If you increase this limit, be aware that the current
#'     implementation pre-allocates the equivalent of a named list for them.}
#'     }
#'
#' @export
#'
nano_init <- function(warn = c("immediate", "deferred", "error", "none")) {

  warn <- match.arg2(warn, c("immediate", "deferred", "error", "none"))
  warn <- switch(warn, 1L, 0L, 2L, -1L)
  if (is.null(getOption("nanonext.original.warn"))) options(nanonext.original.warn = getOption("warn"))
  options(warn = warn)
  invisible(warn)

}

#' Logging Level
#'
#' This function is deprecated.
#'
#' @param level specify a logging level. No longer used.
#'
#' @return Invisible NULL. If the function is called with no arguments,
#'     the logical code of the logging level is returned instead.
#'
#' @details The environment variable 'NANONEXT_LOG' is checked automatically on
#'     package load. If the variable is set incorrectly, the default level
#'     of 'error' is used instead.
#'
#' @keywords internal
#' @export
#'
logging <- function(level = c("keep", "check", "error", "info")) {

  missing(level) && return(.logging.)
  cat("nanonext logging is deprecated and this function can no longer be used\n",
      "logging level can still be set via the environment variable NANONEXT_LOG\n",
      "prior to package load for the time being\n", file = stderr())

}

# nanonext - Limited scope exported functions ----------------------------------

#' @export
#'
.mirai_scm <- function() {

  identical(parent.env(parent.env(parent.frame())), getNamespace("mirai")) ||
    stop("this function is for package internal use only")
  .Call(rnng_scm)

}

# nanonext - Non-exported functions --------------------------------------------

encode <- function(data, mode) {
  switch(mode,
         serialize(object = data, connection = NULL),
         if (is.raw(data)) data else writeBin(object = data, con = raw()))
}

decode <- function(con, mode) {
  switch(mode,
         unserialize(connection = con),
         (r <- readBin(con = con, what = "character", n = length(con)))[r != ""],
         readBin(con = con, what = "complex", n = length(con)),
         readBin(con = con, what = "double", n = length(con)),
         readBin(con = con, what = "integer", n = length(con)),
         readBin(con = con, what = "logical", n = length(con)),
         readBin(con = con, what = "numeric", n = length(con)),
         con)
}

match.arg2 <- function(choice, choices) {
  identical(choice, choices) && return(1L)
  index <- pmatch(choice[1L], choices, nomatch = 0L, duplicates.ok = TRUE)
  index || stop(sprintf("'%s' should be one of %s",
                        deparse(substitute(choice)), paste(choices, collapse = ", ")))
  index
}

loginfo <- function(evt, pkey, pval, skey, sval) {
  cat(sprintf("%s [ %s ] %s: %d | %s: %s\n",
              format.POSIXct(Sys.time()), evt, pkey, pval, skey, sval), file = stdout())
}

