% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_rsquared.R
\name{calc_rsquared}
\alias{calc_rsquared}
\title{Calculate R-Squared.}
\usage{
calc_rsquared(y, yhat, ybar = NULL, return_ss_only = FALSE, threads = 1)
}
\arguments{
\item{y}{The true outcome. This must be a numeric vector, numeric matrix, or
coercible to a sparse matrix of class \code{dgCMatrix}. See 'Details'
below for more information.}

\item{yhat}{The predicted outcome or a list of two matrices whose dot product
makes the predicted outcome. See 'Details' below for more information.}

\item{ybar}{Numeric scalar or vector; the mean of \code{y}. Useful for parallel
computation in batches.}

\item{return_ss_only}{Logical. Do you want to forego calculating R-squared and
only return the sums of squares?}

\item{threads}{Integer number of threads for parallelism; defaults to 1.}
}
\value{
If \code{return_ss_only = FALSE}, \code{calc_rsqured} returns a numeric
    scalar R-squared. If \code{return_ss_only = TRUE}, \code{calc_rsqured}
    returns a vector; the first element is the error sum of squares (SSE) and
    the second element is the total sum of squares (SST). R-squared may then
    be calculated as \code{1 - SSE / SST}.
}
\description{
Calculate R-Squared for univariate or multivariate outcomes.
}
\details{
There is some flexibility in what you can pass as \code{y} and \code{yhat}.
    In general, \code{y} can be a numeric vector, numeric matrix, a sparse
    matrix of class \code{dgCMatrix} from the \code{\link[Matrix]{Matrix}} package,
    or any object that can be coerced into a \code{dgCMatrix}.

    \code{yhat} can be a numeric vector, numeric matrix, or a list of two
    matrices whose dot product has the same dimensionality as \code{y}. If
    \code{yhat} is a list of two matrices you may optionally name them \code{x}
    and \code{w} indicating the order of multiplication (\code{x} left
    multiplies \code{w}). If unnamed or ambiguously named, then it is assumed
    that \code{yhat[[1]]} left multiplies \code{yhat[[2]]}.
}
\note{
On some Linux systems, setting \code{threads} greater than 1 for parallelism
    may introduce some imprecision in the calculation. As of this writing, the
    cause is still under investigation. In the meantime setting \code{threads = 1}
    should fix the issue.

    Setting \code{return_ss_only} to \code{TRUE} is useful for parallel or
    distributed computing for large data sets, particularly when \code{y} is
    a large matrix. However if you do parallel execution you MUST pre-calculate
    'ybar' and pass it to the function. If you do not, SST will be calculated
    based on means of each batch independently. The resulting r-squared will
    be incorrect.

    See example below for parallel computation with \code{\link[furrr]{future_map}}
    from the \code{furr} package.
}
\examples{

# standard r-squared with y and yhat as vectors
f <- stats::lm(mpg ~ cyl + disp + hp + wt, data = datasets::mtcars)

y <- f$model$mpg

yhat <- f$fitted.values

calc_rsquared(y = y, yhat = yhat)

# standard r-squared with y as a matrix and yhat containing 'x' and linear coefficients
s <- summary(f)

x <- cbind(1, as.matrix(f$model[, -1]))

w <- matrix(s$coefficients[, 1], ncol = 1)

calc_rsquared(y = matrix(y, ncol = 1), yhat = list(x, w))

# multivariate r-squared with y and yhat as matrices
calc_rsquared(y = cbind(y, y), yhat = cbind(yhat, yhat))

# multivariate r-squared with yhat as a linear reconstruction of two matrices
calc_rsquared(y = cbind(y, y), yhat = list(x, cbind(w,w)))
}
