\name{mpr}
\alias{mpr}
\alias{print.mpr}
\title{
Fitting a Multi-Parameter Regression (MPR) model.
}
\description{
 Fits a Multi-Parameter Regression (MPR) model using a Newton-type algorithm via the \code{\link{nlm}}
 function.
}
\usage{
mpr(formula, data, family = "Weibull", init, iterlim = 1000, ...)
}
\arguments{
  \item{formula}{
   a two-sided \code{formula} object with the response on the left hand side of the \code{~} operator
   and a \code{list} of one-sided \code{formula} objects on the right hand side (one for each
   regression component in the \code{mpr} model). The response must be a right-censored survival object
   as returned by the \code{Surv} function. See \dQuote{Details} for more information on the struture of
   the formula within the \code{mpr} function as it differs from standard regression models.
}
  \item{data}{
   an optional \code{data.frame} containing the variables in the model. If missing, the variables are 
   taken from the \code{environment} from which \code{mpr} is called.
} 
  \item{family}{
   the name of the parametric distribution to be used in the model. See \code{\link{distributions}} for the 
   list of distributions currently available.
}
  \item{init}{
   an optional vector of initial values for the optimisation routine. If missing, default values
   are used. One may also set \code{init="random"} to randomly generate initial values.
}
  \item{iterlim}{
  a positive integer specifying the maximum number of iterations to be performed before the
  optimisation procedure is terminated. This is supplied to \code{\link{nlm}}.
}
  \item{\dots}{
   additional arguments to be passed to \code{nlm}.
}
}

\details{
 Multi-Parameter Regression (MPR) models are generated by allowing \emph{multiple} distributional parameters
 to depend on covariates, for example, both the scale and shape parameters. This is in contrast to the
 more typical approach where covariates enter a model only through \emph{one} distributional parameter. As 
 these standard models have a single regression component, we may refer to them as Single Parameter
 Regression (SPR) models and, clearly, they are special cases of MPR models. The parameter through
 which covariates enter such SPR models may be referred to as the \dQuote{interest} parameter since it 
 generally has some specific subject-matter importance. However, this standard approach neglects other
 parameters which
 may also be important in describing the phenomenon at hand. The MPR approach generalises the standard
 SPR approach by viewing all distributional parameters as interest parameters in which covariate effects
 can be investigated.

 In the context of survival analysis (currently the focus of the \code{mpr} package), the Weibull model
 is one of the most popular parametric models. Its hazard function is given by
 \deqn{
 h(t) = \lambda \gamma t^{\gamma - 1} }{
 h(t) = \lambda \gamma t^(\gamma - 1) }
 where \eqn{\lambda > 0}, the scale parameter, controls the overall magnitude of \eqn{h(t)} and
 \eqn{\gamma > 0}, the shape parameter, controls its time evolution. In the standard SPR Weibull model,
 \eqn{\lambda} depends on covariates via \eqn{\log \lambda = x^T \beta}{log \lambda = x' \beta} leading to a
 proportional hazards (PH) model. The MPR model generalises this by allowing \emph{both} parameters to
 depend on covariates as follows
 \deqn{\log \lambda = x^T \beta}{log \lambda = x' \beta}
 \deqn{\log \gamma = z^T \alpha}{log \gamma = z' \alpha}
 where \eqn{x} and \eqn{z} are the scale and shape covariate vectors (which may or may not
 contain covariates in common) and \eqn{\beta} and \eqn{\alpha} are the corresponding
 regression coefficients.

 Note that the log-link is used above to ensure positivity of the parameters. More generally, we may
 have
 \deqn{g_1(\lambda) = x^T \beta}{g1(\lambda) = x' \beta}
 \deqn{g_2(\gamma) = z^T \alpha}{g2(\gamma) = z' \alpha}
 where \eqn{g_1(\cdot)}{g1(.)} and \eqn{g_2(\cdot)}{g2(.)} are appropriate link functions.
 The \code{mpr} function does not
 allow the user to alter these link functions but, rather, uses the following default link functions:
 \bold{log-link} (for parameters which must be \emph{positive}) and \bold{identity-link} (for parameters
 which are \emph{unconstrained}). Although the two-parameter Weibull distribution is discussed here
 (due to its popularity), other distributions may have additional shape parameters, for example,
 \deqn{g_3(\rho) = w^T \tau}{g3(\rho) = w' \tau}
 where \eqn{w} and \eqn{\tau} are the vectors of covariates and regression coefficients for this
 additional shape component. See \code{\link{distributions}} for further details on the distributions
 currently available.

 The struture of the \code{formula} within the \code{mpr} function is, for example,
 \code{Surv(time, status) ~ list(~ x1 + x2, ~ x1)} which clearly generalises the typical \code{formula} used in
 standard models (i.e., those with only one regression component) in the sense that the right hand side is a
 \code{list} of one-sided \code{formula} objects. Note the requirement that the \code{~} operator
 precedes each element within the \code{list}. Specifically, the example shown here represents the case
 where the covariates \code{x1} and \code{x2} appear in the first regression component, \eqn{\lambda},
 and the covariate \code{x2} appears in the second regression component, \eqn{\gamma}. If there was a
 third regression component, \eqn{\rho}, then there would be an additional component in the \code{list},
 for example, \code{Surv(time, status) ~ list(~ x1 + x2, ~ x1, ~ x1)}. The \code{mpr} function also accepts
 more typical two-sided \code{formula} objects, such as \code{Surv(time, status) ~ x1 + x2}, which imply that
 the terms on the right hand side appear in \emph{each} of the regression components.
} 

\value{
 \code{mpr} returns an object of class \dQuote{\code{mpr}}.

 The function \code{summary} (i.e., \code{\link{summary.mpr}}) can be used to obtain and print a summary
 of the results. The the generic accessor function \code{coefficients} extracts the \code{list} of
 regression coefficient vectors. One can also apply \code{predict} (i.e., \code{\link{predict.mpr}}) to
 predict various quantites from the fitted \code{mpr} model. A stepwise variable selection procedure has
 been implemented for \code{mpr} models - see \code{\link{stepmpr}}.

 An object of class \code{mpr} is a \code{list} containing the following components: 

  \item{model}{
  a \code{data.frame} containing useful information about the fitted model with the following headings:
  \describe{
    \item{\code{family}}{
    the chosen \code{\link{distribution}}.
    }
    \item{\code{npar}}{
    number of estimated parameters in the fitted model.
    }
    \item{\code{loglike}}{
    value of the log-likelihood.
    }
    \item{\code{aic}}{
    value of the AIC (Akaike Information Criterion).
    }
    \item{\code{bic}}{
    value of the BIC (Bayesian Information Criterion).
    }
    \item{\code{code}}{
    an integer indicating why the Newton optimisation procedure terminated (for more
    details on this stop-code see \code{\link{nlm}}) where, in particular, \dQuote{\code{1}} means
    \dQuote{relative gradient is close to zero}. 
    }
  }
  }
  \item{coefficients}{
  a \code{list} whose elements are named vectors of coefficients (one vector per regression component).
  }
  \item{vcov}{
  the variance-covariance matrix for the estimates.
  }
  \item{gradient}{
  the values of the (negative) score functions from \code{\link{nlm}}.
  }
  \item{ncomp}{
  the number of regression components in the model, i.e., the number of distributional parameters in the 
  underlying \code{\link{distribution}}.
  }
  \item{formula}{
  the \code{formula} supplied.
  }
  \item{xvars}{
  a record of the names of all variables (i.e., covariates) used in fitting.
  }
  \item{xlevels}{
  a record of the levels of any \code{\link{factor}s} (i.e., categorical variables) used in fitting.
  }
  \item{call}{
  the matched call.
  }
}

%%\references{ }

\author{
 Kevin Burke.
}

%%\note{ }

\seealso{
\code{\link{distributions}}, \code{\link{summary.mpr}}, \code{\link{predict.mpr}},
\code{\link{stepmpr}}.
}
\examples{
# Veterans' administration lung cancer data
data(veteran, package="survival")
head(veteran)

# treatment variable, "trt", in scale (lambda) and shape (gamma)
# components of a Weibull model
mpr(Surv(time, status) ~ list(~ trt, ~ trt), data=veteran, family="Weibull")

# same as first model
mpr(Surv(time, status) ~ trt, data=veteran, family="Weibull")

# now with "celltype" also appearing in the scale
mpr(Surv(time, status) ~ list(~ trt + celltype, ~ trt), data=veteran,
    family="Weibull")

# trt in scale only (this is a PH Weibull model)
mpr(Surv(time, status) ~ list(~ trt, ~ 1), data=veteran, family="Weibull")

# trt in all three components (scale and two shape parameters) of a Burr model
mpr(Surv(time, status) ~ list(~ trt, ~ trt, ~ trt), data=veteran,
    family="Burr")

# use of summary
mod1 <- mpr(Surv(time, status) ~ list(~ trt, ~ trt), data=veteran)
summary(mod1)
}
