#' @title Feature Selection via Random Search
#'
#' @name mlr_fselectors_random_search
#'
#' @description
#' Random search randomly draws feature sets.
#'
#' Feature sets are evaluated in batches of size `batch_size`.
#' Larger batches mean we can parallelize more, smaller batches imply a more fine-grained checking of termination criteria.
#'
#' @templateVar id random_search
#' @template section_dictionary_fselectors
#'
#' @section Parameters:
#' \describe{
#' \item{`max_features`}{`integer(1)`\cr
#' Maximum number of features. By default, number of features in [mlr3::Task].}
#' \item{`batch_size`}{`integer(1)`\cr
#' Maximum number of feature sets to try in a batch.}
#' }
#'
#' @source
#' `r format_bib("bergstra_2012")`
#'
#' @export
#' @examples
#' # retrieve task
#' task = tsk("pima")
#'
#' # load learner
#' learner = lrn("classif.rpart")
#'
#' \donttest{
#' # feature selection on the pima indians diabetes data set
#' instance = fselect(
#'   method = "random_search",
#'   task = task,
#'   learner = learner,
#'   resampling = rsmp("holdout"),
#'   measure = msr("classif.ce"),
#'   term_evals = 100
#' )
#'
#' # best performing feature subset
#' instance$result
#'
#' # all evaluated feature subsets
#' as.data.table(instance$archive)
#'
#' # subset the task and fit the final model
#' task$select(instance$result_feature_set)
#' learner$train(task)
#' }
FSelectorRandomSearch = R6Class("FSelectorRandomSearch",
  inherit = FSelector,
  public = list(

    #' @description
    #' Creates a new instance of this [R6][R6::R6Class] class.
    initialize = function() {
      ps = ps(
        "max_features" = p_int(lower = 1),
        "batch_size" = p_int(default = 1, lower = 1)
      )

      ps$values = list(batch_size = 1L)

      super$initialize(
        param_set = ps,
        properties = c("single-crit", "multi-crit"),
        label = "Random Search",
        man = "mlr3fselect::mlr_fselectors_random_search"
      )
    }
  ),

  private = list(
    .optimize = function(inst) {
      pars = self$param_set$values
      feature_names = inst$archive$cols_x
      max_features = pars$max_features %??% length(feature_names)

      repeat {
        X = t(replicate(pars$batch_size, {
          n = sample.int(max_features, 1L)
          x = sample.int(length(feature_names), n)
          replace(logical(length(feature_names)), x, TRUE)
        }))
        colnames(X) = feature_names
        inst$eval_batch(as.data.table(X))
      }
    }
  )
)

mlr_fselectors$add("random_search", FSelectorRandomSearch)
