% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mies_methods.R
\name{mies_init_population}
\alias{mies_init_population}
\title{Initialize MIES Optimization}
\usage{
mies_init_population(
  inst,
  mu,
  initializer = generate_design_random,
  survival_selector = SelectorBest$new()$prime(inst$search_space),
  budget_id = NULL,
  fidelity = NULL,
  fidelity_new_individuals_only = FALSE,
  fidelity_monotonic = TRUE,
  additional_component_sampler = NULL
)
}
\arguments{
\item{inst}{(\code{OptimInstance})\cr
Optimization instance to evaluate.}

\item{mu}{(\code{integer(1)})\cr
Population target size, non-negative integer.}

\item{initializer}{(\code{function})\cr
Function that generates a \code{\link[paradox:Design]{Design}} object, with arguments \code{param_set} and \code{n}, functioning like \code{\link[paradox:generate_design_random]{paradox::generate_design_random}}
or \code{\link[paradox:generate_design_lhs]{paradox::generate_design_lhs}}. Note that \code{\link[paradox:generate_design_grid]{paradox::generate_design_grid}} can not be used and must be wrapped with
a custom function that ensures that only \code{n} individuals are produced. The generated design must correspond to the \code{inst}'s \verb{$search_space}; for
components that are not in the objective's search space, the \code{additional_component_sampler} is used.}

\item{survival_selector}{(\code{\link{Selector}})\cr
Used when the given \code{OptimInstance} already contains more individuals than \code{mu}.\cr
\code{\link{Selector}} operator that selects surviving individuals depending on configuration values
and objective results,  When \code{survival_selector$operate()} is called, then objectives that
are being minimized are multiplied with -1 (through \code{\link{mies_get_fitnesses}}), since \code{\link{Selector}}s always try to maximize fitness.\cr
The \code{\link{Selector}} must be primed on \code{inst$search_space}; this \emph{includes} the "budget" component
when performing multi-fidelity optimization. Default is \code{\link{SelectorBest}}.\cr
The given \code{\link{Selector}} may \emph{not} return duplicates.\cr}

\item{budget_id}{(\code{character(1)} | \code{NULL})\cr
Budget component when doing multi-fidelity optimization. This component of the search space is added
to \code{individuals} according to \code{fidelity}. Should be \code{NULL} when no multi-fidelity optimization is performed (default).}

\item{fidelity}{(\code{atomic(1)} | \code{NULL})\cr
Atomic scalar indicating the value to be assigned to the \code{budget_id} component of offspring.
This value must be \code{NULL} if no multi-fidelity optimization is performed (the default).}

\item{fidelity_new_individuals_only}{(\code{logical(1)})\cr
When \code{fidelity} is not \code{NULL}: Whether to re-evaluate individuals that are already present in \code{inst} should they have a smaller (if \code{fidelity_monotonic} is \code{TRUE}) or different
(if \code{fidelity_monotonic} is \code{FALSE}) value from the one given to \code{fidelity}. Default \code{FALSE}. Ignored when \code{fidelity} is \code{NULL}.}

\item{fidelity_monotonic}{(\code{logical(1)})\cr
Whether to only re-evaluate configurations for which the fidelity would increase. Default \code{TRUE}.
Ignored when \code{fidelity} is \code{NULL} or when \code{fidelity_new_individuals_only} is \code{TRUE}.}

\item{additional_component_sampler}{(\code{\link[paradox:Sampler]{Sampler}} | \code{NULL})\cr
\code{\link[paradox:Sampler]{Sampler}} for components of individuals that are not part of \code{inst}'s \verb{$search_space}. These components
are never used for performance evaluation, but they may be useful for self-adaptive \code{\link{OperatorCombination}}s. See the description
of \code{\link[=mies_prime_operators]{mies_prime_operators()}} on how operators need to be primed to respect additional components.\cr
It is possible that \code{additional_component_sampler} is used for \emph{more} rows than \code{initializer}, which happens
when the \code{inst}'s \verb{$archive} contains prior evaluations that are alive, but does not contain columns pertaining to additional columns,
or contains \emph{all} these columns but there are rows that are \code{NA} valued. If only \emph{some} of the columns are present, or if all these columns
are present but there are rows that are only \code{NA} valued for some columns, then an error is thrown.\cr
Default is \code{NULL}: no additional components.}
}
\value{
\link{invisible} \code{OptimInstance}: the input
instance, modified by-reference.
}
\description{
Set up an \code{OptimInstance} for MIES optimization.
This adds the \code{dob} and \code{eol} columns to the instance's archive, and makes sure there are at least \code{mu} survivors
(i.e. entries with \code{eol} set to \code{NA}) present. If there are already \verb{>= mu} prior evaluations present, then the last
\code{mu} of these remain alive (the other's \code{eol} set to 0); otherwise, up to \code{mu} new randomly sampled configurations
are evaluated and added to the archive and have \code{eol} set to \code{NA}.
}
\examples{
library("bbotk")
lgr::threshold("warn")

# Define the objective to optimize
objective <- ObjectiveRFun$new(
  fun = function(xs) {
    z <- exp(-xs$x^2 - xs$y^2) + 2 * exp(-(2 - xs$x)^2 - (2 - xs$y)^2)
    list(Obj = z)
  },
  domain = ps(x = p_dbl(-2, 4), y = p_dbl(-2, 4)),
  codomain = ps(Obj = p_dbl(tags = "maximize"))
)

# Get a new OptimInstance
oi <- OptimInstanceSingleCrit$new(objective,
  terminator = trm("evals", n_evals = 100)
)

mies_init_population(inst = oi, mu = 3)

# 3 evaluations, archive contains 'dob' and 'eol'
oi$archive

###
# Advanced demo, making use of additional components and fidelity
##

# Get a new OptimInstance
oi <- OptimInstanceSingleCrit$new(objective,
  terminator = trm("evals", n_evals = 100)
)

mies_init_population(inst = oi, mu = 3, budget_id = "y", fidelity = 2,
  additional_component_sampler = Sampler1DRfun$new(
    param = ps(additional = p_dbl(-1, 1)), rfun = function(n) rep(-1, n)
  )
)

# 3 evaluations. We also have 'additional', sampled from rfun (always -1),
# which is ignored by the objective. Besides, we have "y", which is 2,
# according to 'fidelity'.
oi$archive

}
\seealso{
Other mies building blocks: 
\code{\link{mies_evaluate_offspring}()},
\code{\link{mies_generate_offspring}()},
\code{\link{mies_get_fitnesses}()},
\code{\link{mies_select_from_archive}()},
\code{\link{mies_step_fidelity}()},
\code{\link{mies_survival_comma}()},
\code{\link{mies_survival_plus}()}
}
\concept{mies building blocks}
