% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bend.R
\name{bend}
\alias{bend}
\title{Matrix bending to positive-definite}
\usage{
bend(
  inmat,
  wtmat,
  reciprocal = FALSE,
  max.iter = 10000,
  small.positive = 1e-04,
  method = "hj"
)
}
\arguments{
\item{inmat}{: The \code{matrix} to be bent.}

\item{wtmat}{: The weight \code{matrix} for weighted bending. If no input is provided, the unweighted method (default) is used.}

\item{reciprocal}{: If \code{TRUE}, reciprocal of the weighting factors are used. If no input is provided, default = \code{FALSE}.}

\item{max.iter}{: Maximum number of iterations. If no input is provided, default = 10000.}

\item{small.positive}{: Eigenvalues smaller than this value are replaced with this value. If no input is provided, default = 0.0001.}

\item{method}{: \code{"hj"} (Jorjani et al., 2003) or \code{"lrs"} (Schaeffer, 2014), default = \code{"hj"}}
}
\value{
bent : The bent \code{matrix}.

init.ev : Eigenvalues of the initial (\code{inmat}) matrix.

final.ev : Eigenvalues of the \code{bent} matrix.

min.dev : \code{min(bent - inmat)}.

max.dev : \code{max(bent - inmat)}.

loc.min.dev : Location (indices) of \code{min.dev} element.

loc.max.dev : Location (indices) of \code{max.dev} element.

ave.dev : Average deviation (\code{bent - inmat}) of the upper triangle elements (excluding diagonal elements for correlation matrices).

AAD : Average absolute deviation of the upper triangle elements (excluding diagonal elements for correlation matrices) of \code{bent} and \code{inmat}.

Cor : Correlation between the upper triangle elements (excluding diagonal elements for correlation matrices) of \code{bent} and \code{inmat}.

RMSD : Root of mean squared deviation of the upper triangle elements (excluding diagonal elements for correlation matrices) of \code{bent} and \code{inmat}.

w_gt_0 : Number of weight elements greater than 0, in the upper triangle of \code{wtmat} (for weighted bending).

wAAD : Weighted \code{AAD} (for weighted bending).

wCor : Weighted \code{Cor} (for weighted bending).

wRMSD : Weighted \code{RMSD} (for weighted bending).
}
\description{
Bending a symmetric non-positive-definite matrix to positive-definite, using weighted or unweighted methods.
}
\examples{
# Test data
V = matrix(nrow=5, ncol=5, c( # matrix to be bent
100,  95,  80,  40,  40,
 95, 100,  95,  80,  40,
 80,  95, 100,  95,  80,
 40,  80,  95, 100,  95,
 40,  40,  80,  95, 100))
W = matrix(nrow=5, ncol=5, c( # matrix of weights
1000,  500,   20,   50,  200,
 500, 1000,  500,    5,   50,
  20,  500, 1000,   20,   20,
  50,    5,   20, 1000,  200,
 200,   50,   20,  200, 1000))

# Example 1: Unweighted bending
bend(V)
## The default method (Jojani et al. 2003) is used.

# Example 2: Weighted bending using reciprocal of the weighting factors
bend(inmat=V, wtmat=W, reciprocal=TRUE)

# Example 3: Bending with fixed elements
## Assume we want to keep V[1:2, 1:2] constant.
W2 = W; W2[1:2, 1:2] = 0
bend(inmat=V, wtmat=W2, reciprocal=TRUE)

# Example 4: Bending a correlation matrix
V2 = cov2cor(V)
bend(V2, W, reciprocal=TRUE)

# Example 5: Bending using the method of Schaeffer (2014)
bend(inmat=V, method="lrs")

# Example 6: Bending a correlation matrix using the method of Schaeffer (2014)
bend(V2, method="lrs")

# Example 7: Bending the same correlation matrix using a weighted development of Schaeffer (2014)
bend(V2, W, reciprocal=TRUE, method="lrs")

# Example 8: Bending a covariance matrix using a weighted development of Schaeffer (2014)
bend(V, W, reciprocal=TRUE, method="lrs")

}
