% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dydx.R
\name{dydx}
\alias{dydx}
\alias{dydx.default}
\alias{dydx.factor}
\alias{dydx.ordered}
\alias{dydx.logical}
\title{Marginal Effect of a Given Variable}
\usage{
dydx(data, model, variable, ...)

\method{dydx}{default}(data, model, variable, type = c("response", "link"),
  change = c("dydx", "minmax", "iqr", "sd"), eps = 1e-07,
  as.data.frame = TRUE, ...)

\method{dydx}{factor}(data, model, variable, type = c("response", "link"),
  fwrap = FALSE, as.data.frame = TRUE, ...)

\method{dydx}{ordered}(data, model, variable, type = c("response", "link"),
  fwrap = FALSE, as.data.frame = TRUE, ...)

\method{dydx}{logical}(data, model, variable, type = c("response", "link"),
  as.data.frame = TRUE, ...)
}
\arguments{
\item{data}{The dataset on which to to calculate \eqn{\hat{y}}.}

\item{model}{The model object to pass to \code{\link[prediction]{prediction}}.}

\item{variable}{A character string specifying the variable to calculate the derivative or discrete change for.}

\item{\dots}{Ignored.}

\item{type}{The type of prediction. Default is \dQuote{response}.}

\item{change}{For numeric variables, a character string specifying the type of change to express. The default is the numerical approximation of the derivative. Alternative values are occasionally desired quantities: \dQuote{minmax} (the discrete change moving from \code{min(x)} to \code{max(x)}), \dQuote{iqr} (the move from the 1st quartile to 3rd quartile of \code{x}), or \dQuote{sd} (the change from \code{mean(x) - sd(x)} to \code{mean(x) + sd(x)}), or a two-element numeric vector expressing values of the variable to calculate the prediction for (and difference the associated predictions).}

\item{eps}{If \code{change == "dydx"} (the default), the value of the step \eqn{\epsilon} to use in calculation of the numerical derivative for numeric variables.}

\item{as.data.frame}{A logical indicating whether to return a data frame (the default) or a matrix.}

\item{fwrap}{A logical specifying how to name factor columns in the response.}
}
\value{
A data frame, typically with one column unless the variable is a factor with more than two levels. The names of the marginal effect columns begin with \dQuote{dydx_} to distinguish them from the substantive variables of the same names.
}
\description{
Differentiate an Estimated Model Function with Respect to One Variable, or calculate a discrete difference (\dQuote{first difference}) as appropriate.
}
\details{
These functions provide a simple interface to the calculation of marginal effects for specific variables used in a model, and are the workhorse functions called internally by \code{\link{marginal_effects}}.

\code{dydx} is an S3 generic with classes implemented for specific variable types. S3 method dispatch, somewhat atypically, is based upon the class of \code{data[[variable]]}.

For numeric (and integer) variables, the method calculates an instantaneous marginal effect using a simple \dQuote{central difference} numerical differentiation:
\deqn{\frac{f(x + \frac{1}{2}h) - f(x - \frac{1}{2}h)}{dh}}{(f(x + 0.5h) - f(x - 0.5h))/(2h)}, where (\eqn{h = \max(|x|, 1) \sqrt{\epsilon}}{h = max(|x|, 1)sqrt(epsilon)} and the value of \eqn{\epsilon}{epsilon} is given by argument \code{eps}. This procedure is subject to change in the future.

For factor variables (or character variables, which are implicitly coerced to factors by modelling functions), discrete first-differences in predicted outcomes are reported instead (i.e., change in predicted outcome when factor is set to a given level minus the predicted outcome when the factor is set to its baseline level). These are sometimes called \dQuote{partial effects}. If you want to use numerical differentiation for factor variables (which you probably do not want to do), enter them into the original modelling function as numeric values rather than factors.

For ordered factor variables, the same approach as factors is used. This may contradict the output of modelling function summaries, which rely on \code{options("contrasts")} to determine the contrasts to use (the default being \code{\link[stats]{contr.poly}} rather than \code{\link[stats]{contr.treatment}}, the latter being used normally for unordered factors).

For logical variables, the same approach as factors is used, but always moving from \code{FALSE} to \code{TRUE}.
}
\examples{
require("datasets")
x <- lm(mpg ~ cyl * hp + wt, data = head(mtcars))
# marginal effect (numerical derivative)
dydx(head(mtcars), x, "hp")

# other discrete differences
## change from min(mtcars$hp) to max(mtcars$hp)
dydx(head(mtcars), x, "hp", change = "minmax")
## change from 1st quartile to 3rd quartile
dydx(head(mtcars), x, "hp", change = "iqr")
## change from mean(mtcars$hp) +/- sd(mtcars$hp)
dydx(head(mtcars), x, "hp", change = "sd")
## change between arbitrary values of mtcars$hp
dydx(head(mtcars), x, "hp", change = c(75,150))

# factor variables
mtcars[["cyl"]] <- factor(mtcars$cyl)
x <- lm(mpg ~ cyl, data = head(mtcars))
dydx(head(mtcars), x, "cyl")

}
\references{
Miranda, Mario J. and Paul L. Fackler. 2002. \emph{Applied Computational Economics and Finance}. p. 103.

  Greene, William H. 2012. \emph{Econometric Analysis}. 7th edition. pp. 733--741.

  Cameron, A. Colin and Pravin K. Trivedi. 2010. \emph{Microeconometric Using Stata}. Revised edition. pp. 106--108, 343--356, 476--478.
}
\seealso{
\code{\link{marginal_effects}}, \code{\link{margins}}
}
