#' Gene tracks using 'plotly'
#' 
#' Plot gene annotation tracks from `ensembldb` data using `plotly`.
#' 
#' @details This function can used to plot gene annotation tracks on their own.
#' @param locus Object of class 'locus' generated by [locus()].
#' @param filter_gene_name Vector of gene names to display.
#' @param filter_gene_biotype Vector of gene biotypes to be filtered. Use
#' [ensembldb::listGenebiotypes()] to display possible biotypes. For example, 
#' `ensembldb::listGenebiotypes(EnsDb.Hsapiens.v75)`
#' @param cex.text Font size for gene text.
#' @param maxrows Specifies maximum number of rows to display in gene 
#' annotation panel.
#' @param xlab Title for x axis. Defaults to chromosome `seqname` specified 
#' in `locus`.
#' @param gene_col Colour for gene lines.
#' @param exon_col Fill colour for exons.
#' @param exon_border Border line colour outlining exons. Set to `NA` for no 
#' border.
#' @return A 'plotly' plotting object showing gene tracks.
#' @examples
#' if(require(EnsDb.Hsapiens.v75)) {
#' library(EnsDb.Hsapiens.v75)
#' data(SLE_gwas_sub)
#' loc <- locus(SLE_gwas_sub, gene = 'UBE2L3', flank = 1e5,
#'              ens_db = "EnsDb.Hsapiens.v75")
#' genetrack_ly(loc)
#' }
#' @importFrom plotly plot_ly plotly_empty add_segments add_text %>%
#' @export

genetrack_ly <- function(locus,
                         filter_gene_name = NULL,
                         filter_gene_biotype = NULL,
                         cex.text = 0.7,
                         gene_col = 'blue4',
                         exon_col = 'blue4',
                         exon_border = 'blue4',
                         maxrows = 8,
                         xlab = NULL) {
  if (!inherits(locus, "locus")) stop("Object of class 'locus' required")
  TX <- locus$TX
  EX <- as.data.frame(locus$EX)
  xrange <- locus$xrange
  if (!is.null(filter_gene_name)) {
    TX <- TX[TX$gene_name %in% filter_gene_name, ]
  }
  if (!is.null(filter_gene_biotype)) {
    TX <- TX[TX$gene_biotype %in% filter_gene_biotype, ]
  }
  if (nrow(TX) == 0) {
    message('No genes to plot')
    return(plotly_empty())
  }
  if (is.null(xlab)) xlab <- paste("Chromosome", locus$seqname, "(Mb)")
  
  TX <- mapRow(TX, xlim = xrange, cex.text = cex.text)
  maxrows <- if (is.null(maxrows)) max(TX$row) else min(c(max(TX$row), maxrows))
  if (max(TX$row) > maxrows) message(max(TX$row), " tracks needed to show all genes")
  TX <- TX[TX$row <= maxrows, ]
  EX <- EX[EX$gene_id %in% TX$gene_id, ]
  
  gene_col <- col2hex(gene_col)
  exon_col <- col2hex(exon_col)
  exon_border <- col2hex(exon_border)
  EX$row <- TX$row[match(EX$gene_id, TX$gene_id)]
  
  EX[, c('start', 'end')] <- EX[, c('start', 'end')] / 1e6
  shapes <- lapply(seq_len(nrow(EX)), function(i) {
    list(type = "rect", fillcolor = exon_col, line = list(color = exon_border,
                                                          width = 0.5),
         x0 = EX$start[i], x1 = EX$end[i], xref = "x",
         y0 = -EX$row[i] - 0.15, y1 = -EX$row[i] + 0.15, yref = "y")
  })
  TX$tx <- rowMeans(TX[, c('start', 'end')])
  TX$ty <- -TX$row + 0.4
  TX[, c('start', 'end', 'tx')] <- TX[, c('start', 'end', 'tx')] / 1e6
  
  xlim <- xrange / 1e6
  xext <- diff(xlim) * 0.05
  xlim <- xlim + c(-xext, xext)
  tfilter <- TX$tmin > (xrange[1] - diff(xrange) * 0.04) & 
             (TX$tmax < xrange[2] + diff(xrange) * 0.04)
  TX$gene_name2 <- TX$gene_name
  TX$gene_name2[!tfilter] <- NA
  
  hovertext <- paste0(TX$gene_name,
                      "<br>Gene ID: ", TX$gene_id,
                      "<br>Biotype: ", TX$gene_biotype,
                      "<br>Start: ", TX$start * 1e6,
                      "<br>End: ", TX$end * 1e6)
  
  plot_ly(TX) %>%
    add_segments(x = ~start, y = ~-row,
                 xend = ~end, yend = ~-row,
                 color = I(gene_col),
                 text = hovertext, hoverinfo = 'text',
                 showlegend = FALSE) %>%
    add_text(x = ~tx, y = ~ty, text = ~gene_name2,
             textfont = list(size = 15 * cex.text),
             showlegend = FALSE, hoverinfo = 'none') %>%
    plotly::layout(shapes = shapes,
                   xaxis = list(title = xlab, showgrid = FALSE, showline = TRUE,
                                color = 'black', ticklen = 5,
                                range = as.list(xlim)),
                   yaxis = list(title = "", showgrid = FALSE, 
                                showticklabels = FALSE)) %>%
    plotly::config(displaylogo = FALSE)
}


#' @importFrom grDevices col2rgb rgb

col2hex <- function(cname) {
  colMat <- col2rgb(cname)
  rgb(red = colMat[1, ]/255, green = colMat[2, ]/255, blue = colMat[3, ]/255)
}
