\name{felm}
\alias{felm}
\encoding{UTF-8}
\title{Fit a linear model with multiple group fixed effects}
\description{
'felm' is used to fit linear models with multiple group fixed effects,
similarly to lm.  It uses the Method of Alternating projections to
sweep out multiple group effects from the normal equations before
estimating the remaining coefficients with OLS.

This function is intended for use with large datasets with multiple group
effects of large cardinality.  If dummy-encoding the group effects
results in a manageable number of coefficients, you are probably better
off by using \code{\link{lm}}.
}

\usage{ felm(formula, data, exactDOF=FALSE, subset, na.action, contrasts=NULL, weights=NULL,...) }

\arguments{
  \item{formula}{an object of class '"formula"' (or one that can be
  coerced to that class): a symbolic description of the model to be
  fitted. Similarly to 'lm'.  See Details.}

  \item{data}{a data frame containing the variables of the model.}

  \item{exactDOF}{logical. If more than two factors, the degrees
        of freedom used to scale the covariance matrix (and the standard
        errors) is normally estimated. Setting \code{exactDOF=TRUE}
        causes \code{felm} to attempt to compute it, but this may fail if there are
        too many levels in the factors.  \code{exactDOF='rM'} will use
        the exact method in \code{Matrix::rankMatrix()}, but this is
        slower. If neither of these methods works, it is possible to
	specify \code{exactDOF='mc'}, which utilizes a Monte-Carlo method to
        estimate the expectation E(x' P x) = tr(P), the trace of a certain projection,
	a method which may be more accurate than the default guess.

	If the degrees of freedom for some reason are known, they can be
        specified like \code{exactDOF=342772}.} 

 \item{subset}{an optional vector specifying a subset of observations to
   be used in the fitting process.}
 
 \item{na.action}{a function which indicates what should happen when the
        data contain \code{NA}s.  The default is set by the
        \code{na.action} setting of \code{options}, and is
        \code{na.fail} if that is unset.  The 'factory-fresh' default is
        \code{na.omit}.  Another possible value is \code{NULL}, no
        action. \code{na.exclude} is currently not supported.}

  \item{contrasts}{an optional list. See the \code{contrasts.arg} of \code{model.matrix.default}.}

  \item{weights}{an optional vector of weights to be used in the fitting
              process.  Should be ‘NULL’ or a numeric vector.  If
              non-NULL, weighted least squares is used with weights
              \code{weights} (that is, minimizing \code{sum(w*e^2)}); otherwise
              ordinary least squares is used.}

  \item{...}{other arguments.
    \itemize{

      \item \code{keepX} logical. To include a copy of the expanded data
      matrix in the return value, as needed by \code{\link{bccorr}} and
      \code{\link{fevcov}} for proper limited mobility bias correction.

      \item \code{keepCX} logical. Keep a copy of the centred expanded
      data matrix in the return value. As list elements \code{cX} for
      the explanatory variables, and \code{cY} for the outcome.

      \item \code{nostats} logical. Don't include covariance matrices in
      the output, just the estimated coefficients and various
      descriptive information. For IV, \code{nostats} can be a logical
      vector of length 2, with the last value being used for the 1st
      stages.

      \item \code{kclass} character. For use with instrumental
      variables. Use a k-class estimator rather than 2SLS/IV. Currently,
      the values \code{'nagar', 'b2sls', 'mb2sls', 'liml'} are accepted,
      where the names are from \cite{Kolesár et al (2014)}, as well as
      a numeric value for the 'k' in k-class. With \code{kclass='liml'},
      \code{felm} also accepts the argument \code{fuller=<numeric>}, for
      using a Füller adjustment of the liml-estimator.

      \item \code{Nboot, bootexpr, bootcluster} Since \code{felm} has
      quite a bit of overhead in the creation of the model matrix, if
      one wants confidence intervals for some function of the estimated
      parameters, it is possible to bootstrap internally in \code{felm}.
      That is, the model matrix is resampled \code{Nboot} times and
      estimated, and the \code{bootexpr} is evaluated inside an
      \code{sapply}. The estimated coefficients and the left hand
      side(s) are available by name. Any right hand side variable
      \code{x} is available by the name \code{var.x}.  The
      \code{"felm"}-object for each estimation is available as
      \code{est}. If a \code{bootcluster} is specified as a factor,
      entire levels are resampled. \code{bootcluster} can also be a
      function with no arguments, it should return a vector of integers,
      the rows to use in the sample. \code{bootexpr} should be an
      expression, e.g. like \code{quote(x/x2 * abs(x3)/mean(y))}.
      It could be wise to specify \code{nostats=TRUE} when
      bootstrapping, unless the covariance matrices are needed in the
      bootstrap. 

      \item \code{iv, clustervar} deprecated.  These arguments will be
      removed at a later time, but are still supported in this
      field. Users are \emph{STRONGLY} encouraged to use multipart
      formulas instead.  In particular, not all functionality is
      supported with the deprecated syntax; iv-estimations actually run
      a lot faster if multipart formulas are used, due to new algorithms
      which I didn't bother to shoehorn in place for the deprecated
      syntax.

    }
  }
}
\value{
  \code{felm} returns an object of \code{class} \code{"felm"}.  It is
  quite similar to an \code{"lm"} object, but not entirely compatible.

  The generic \code{summary}-method will yield a summary which may be \code{print}'ed.
  The object has some resemblance to an \code{'lm'} object, and some
  postprocessing methods designed for \code{lm} may happen to work. It
  may however be necessary to coerce the object to succeed with this.

  The \code{"felm"} object is a list containing the following fields:

  \item{coefficients}{a numerical vector. The estimated coefficients.}
  \item{N}{an integer. The number of observations}
  \item{p}{an integer. The total number of coefficients, including those
    projected out.}
  \item{response}{a numerical vector. The response vector.}
  \item{fitted.values}{a numerical vector. The fitted values.}

  \item{residuals}{a numerical vector. The residuals of the full
  system, with dummies.  For IV-estimations, this is the residuals
  when the original endogenous variables are used, not their predictions
  from the 1st stage.}

  \item{r.residuals}{a numerical vector. Reduced residuals, i.e. the residuals resulting from
  predicting \emph{without} the dummies.}

  \item{iv.residuals}{numerical vector. When using instrumental
  variables, residuals from 2. stage, i.e. when predicting with the
  predicted endogenous variables from the 1st stage.}

  \item{weights}{numeric. The square root of the argument \code{weights}.}

  \item{cfactor}{factor of length N. The factor describing the connected
    components of the two first terms in the second part of the model formula.}

  \item{vcv}{a matrix. The variance-covariance matrix.}

  \item{fe}{list of factors. A list of the terms in the second part of
  the model formula.}


  \item{stage1}{The '\code{felm}' objects for the IV 1st stage, if
    used. The 1st stage has multiple left hand sides if there are more
    than one instrumented variable.}
  
  \item{iv1fstat}{list of numerical vectors. For IV 1st stage, F-value for
    excluded instruments, the number of parameters in restricted model
    and in the unrestricted model.}
    
  \item{X}{matrix. The expanded data matrix, i.e. from the first part of
  the formula. To save memory with large datasets, it is only included if
  \code{felm(keepX=TRUE)} is specified.  Must be included if
  \code{\link{bccorr}} or \code{\link{fevcov}} is to be used for correcting limited mobility bias.
  }

  \item{cX, cY}{matrix. The centred expanded data matrix. Only included if
    \code{felm(keepCX=TRUE)}.
  }

  \item{boot}{The result of a \code{replicate} applied to the
    \code{bootexpr} (if used).}
}

\details{
The formula specification is a response variable followed by a four part
formula. The first part consists of ordinary covariates, the second part
consists of factors to be projected out. The third part is an
IV-specification. The fourth part is a cluster specification for the
standard errors.  I.e. something like \code{y ~ x1 + x2 | f1 + f2 |
  (Q|W ~ x3+x4) | clu1 + clu2} where \code{y} is the response,
\code{x1,x2} are ordinary covariates, \code{f1,f2} are factors to be
projected out, \code{Q} and \code{W} are covariates which are
instrumented by \code{x3} and \code{x4}, and \code{clu1,clu2} are
factors to be used for computing cluster robust standard errors.
Parts that are not used should be specified as \code{0}, except if it's
at the end of the formula, where they can be omitted.  The parentheses
are needed in the third part since \code{|} has higher precedence than \code{~}.
As of lfe version 2.0, multiple left hand sides like
\code{y|w|x ~ x1 + x2 |f1+f2|...} are allowed.

Interactions between a covariate \code{x} and a factor \code{f} can be
projected out with the syntax \code{x:f}. 
The terms in the second and fourth parts are not treated as 
ordinary formulas, in particular it is not possible with things like
\code{y ~ x1 | x*f}, rather one would specify \code{y ~ x1 + x | x:f + f}.
Note that \code{f:x} also works, since R's parser does not keep the
order.  This means that in interactions, the factor \emph{must} be a
factor, whereas a non-interacted factor will be coerced to a
factor. I.e. in \code{y ~ x1 | x:f1 + f2}, the \code{f1} must be a
factor, whereas it will work as expected if \code{f2} is an integer vector.

In older versions of \pkg{lfe} the syntax was \code{felm(y ~ x1 + x2 + G(f1)
+ G(f2), iv=list(Q ~ x3+x4, W ~ x3+x4),
clustervar=c('clu1','clu2'))}. This syntax still works, but yields a
warning. Users are \emph{strongly} encouraged to change to the new
multipart formula syntax.  The old syntax will be removed at a later time.

The standard errors are adjusted for the reduced degrees of freedom
coming from the dummies which are implicitly present.  In the case of
two factors, the exact number of implicit dummies is easy to compute.  If there
are more factors, the number of dummies is estimated by assuming there's
one reference-level for each factor, this may be a slight over-estimation,
leading to slightly too large standard errors. Setting \code{exactDOF='rM'}
computes the exact degrees of freedom with \code{rankMatrix()} in package \pkg{Matrix}.
Note that version 1.1-0 of \pkg{Matrix} has a bug in \code{rankMatrix()}
for sparse matrices which may cause it to return the wrong value. A fix is underway.

For the iv-part of the formula, it is only necessary to include the instruments on the
right hand side.  The other explanatory covariates, from the first and
second part of \code{formula}, are added automatically
in the first stage regression.  See the examples. 

The \code{contrasts} argument is similar to the one in \code{lm()}, it
is used for factors in the first part of the formula. The factors in the
second part are analyzed as part of a possible subsequent \code{getfe()} call.

The old syntax with a single part formula with the \code{G()} syntax for the factors to transform
away is still supported, as well as the \code{clustervar} and \code{iv}
arguments, but users are encouraged to move to the new multi part
formulas as described here.  The \code{clustervar}
and \code{iv} arguments have been moved to the \code{...} argument list.
They will be removed in some future update.

}

\note{

  Side effect: If \code{data} is an object of class \code{"pdata.frame"}
  (from the \pkg{plm} package), the \pkg{plm} namespace is loaded if
  available, and \code{data} is coerced to a \code{"data.frame"} with
  \code{as.data.frame} which dispatches to a \pkg{plm} method.  This
  ensures that transformations like \code{diff} and \code{lag} from
  \pkg{plm} works as expected, but it also incurs an additional copy of
  the \code{data}, and the \pkg{plm} namespace remains loaded after
  \code{felm} returns.  When working with \code{"pdata.frame"}s, this is
  what is usually wanted anyway.

  For technical reasons, when running IV-estimations, the data frame supplied in the \code{data}
  argument to \code{felm}, should \emph{not} contain variables with
  names ending in \code{'(fit)'}.  Variables with such names are used
  internally by \code{felm}, and may then accidentally be looked up in the
  data frame instead of the local environment where they are
  defined. 
}

\seealso{
  \code{\link{getfe}}
  \code{\link{summary.felm}}
  \code{\link{condfstat}}
  \code{\link{waldtest}}
}

\examples{
oldopts <- options(lfe.threads=1)
## create covariates
x <- rnorm(1000)
x2 <- rnorm(length(x))

## individual and firm
id <- factor(sample(20,length(x),replace=TRUE))
firm <- factor(sample(13,length(x),replace=TRUE))

## effects for them
id.eff <- rnorm(nlevels(id))
firm.eff <- rnorm(nlevels(firm))

## left hand side
u <- rnorm(length(x))
y <- x + 0.5*x2 + id.eff[id] + firm.eff[firm] + u

## estimate and print result
est <- felm(y ~ x+x2| id + firm)
summary(est)
\dontrun{
## compare with lm
summary(lm(y ~ x + x2 + id + firm-1))
}

# make an example with 'reverse causation'
# Q and W are instrumented by x3 and the factor x4. Report robust s.e.
x3 <- rnorm(length(x))
x4 <- sample(12,length(x),replace=TRUE)

Q <- 0.3*x3 + x + 0.2*x2 + id.eff[id] + 0.3*log(x4) - 0.3*y + rnorm(length(x),sd=0.3)
W <- 0.7*x3 - 2*x + 0.1*x2 - 0.7*id.eff[id] + 0.8*cos(x4) - 0.2*y+ rnorm(length(x),sd=0.6)

# add them to the outcome
y <- y + Q + W

ivest <- felm(y ~ x + x2 | id+firm | (Q|W ~x3+factor(x4)))
summary(ivest,robust=TRUE)
condfstat(ivest)
\dontrun{
# compare with the not instrumented fit:
summary(felm(y ~ x + x2 +Q + W |id+firm))
}
options(oldopts)
}

\references{
Cameron, A.C., J.B. Gelbach and  D.L. Miller (2011) \cite{Robust
  inference with multiway clustering}, Journal of Business & Economic
Statistics 29 (2011), no. 2, 238--249.
\url{http://dx.doi.org/10.1198/jbes.2010.07136}

Kolesár, M.,  R. Chetty, J. Friedman, E. Glaeser, and G.W. Imbens (2014)
\cite{Identification and Inference with Many Invalid Instruments},
Journal of Business & Economic Statistics (to appear).
\url{http://dx.doi.org/10.1080/07350015.2014.978175}
}
