#pragma once

#include "ldt_base.h"
#include "matrix.h"
#include <functional>
#include <iostream>
#include <iterator>
#include <vector>

namespace ldt {

/// @brief Options of LMBFGS optimization
class LDT_EXPORT LimitedMemoryBfgsbOptions {
public:
  /// @brief initializes the class
  LimitedMemoryBfgsbOptions(){};

  /// @brief Maximum corrections. default is 5.
  Ti mMaxCorrections = 5;

  /// @brief Maximum number of iterations. default is 100.
  Ti IterationMax = 100;

  /// @brief Printing option (not implemented)
  /// less than 0:         no report is generated;
  /// 0                    report only one line at the last iteration;
  /// in (0,99)            report also f and |proj g| every iprint iterations;
  /// 99                   report details of every iteration except n-vectors;
  /// 100                  report also the changes of active set and final x;
  /// larger than 100      report details of every iteration including x and g;
  Ti IterationPrint = 0;

  /// @brief A condition to stop the iteration. The iteration will stop when
  /// (f^k - f^{k+1})/max{|f^k|,|f^{k+1}|,1} .less than. Factor*epsmch
  /// where epsmch is the machine precision, which is automatically generated by
  /// the code. Typical values for Factor: 1e12 for low accuracy; 1e7 for
  /// moderate accuracy; 1e1 for extremely high accuracy. default is 1e7
  Tv Factor = 1e7;

  /// @brief A condition for stopping the iteration. The iteration will stop
  /// when max{|pg_i | i = 1, ..., n} .less than. ProjectedGradientTol where
  /// pg_i is the ith component of the projected gradient. default is 0.
  Tv ProjectedGradientTol = 0;
};

/// @brief Limited Memory Broyden–Fletcher–Goldfarb–Shanno (Bounded)
/// optimization
class LDT_EXPORT LimitedMemoryBFGSB {

private:
  Ti mN = 0;

public:
  /// @brief Options for optimization
  LimitedMemoryBfgsbOptions Options;

  /// @brief Gets the required size of the storage array
  Ti StorageSize = 0;

  /// @brief Gets the required size of the work array (real)
  Ti WorkSize = 0;

  /// @brief Gets the required size of the work array (integer)
  Ti WorkSizeI = 0;

  /// @brief Gets the number of current iteration in \ref Minimize
  Ti Iteration = 0;

  /// @brief Gets the value of the function in \ref Minimize
  Tv FunctionValue = NAN;

  /// @brief Gradient vector in \ref Minimize
  Matrix<Tv> GradientVector;

  /// A reference to the given start point (contains the solution after
  /// optimization)
  Matrix<Tv> *Xstar = nullptr;

  /// @brief Shows why the iteration stopped. see the lbfgsb.h function
  Ti Task = 0;

  /// @brief initializes the class
  LimitedMemoryBFGSB(){};

  /// @brief Initializes a new instance of the class
  /// @param n Number of variables in the objective function
  /// @param maxCorrections Maximum number of corrections. Default is 5.
  LimitedMemoryBFGSB(Ti n, Ti maxCorrections = 5);

  /// @brief Finds the minimum
  /// @param function Objective function
  /// @param gradient Gradient of the objective function
  /// @param startPoint Starting point
  /// @param storage Storage array of size \ref StorageSize
  /// @param work Work array of size \ref WorkSize (real)
  /// @param lower Lower bounds
  /// @param upper Upper bounds
  void Minimize(
      const std::function<Tv(const Matrix<Tv> &)> &function,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &gradient,
      Matrix<Tv> &startPoint, Tv *storage, Tv *work,
      Matrix<Tv> *lower = nullptr, Matrix<Tv> *upper = nullptr);

  /// @brief Similar to \ref Minimize, with more options
  /// @param function
  /// @param gradient
  /// @param startPoint
  /// @param storage
  /// @param work
  /// @param workI Work array of size \ref WorkSizeI (integer)
  /// @param lower
  /// @param upper
  void Minimize0(
      const std::function<Tv(const Matrix<Tv> &)> &function,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &gradient,
      Matrix<Tv> &startPoint, Tv *storage, Tv *work, Ti *workI,
      Matrix<Tv> *lower = nullptr, Matrix<Tv> *upper = nullptr);

  /// @brief Similar to \ref Minimize0 with more options
  /// @param function
  /// @param gradient
  /// @param startPoint
  /// @param storage
  /// @param work
  /// @param iWORK
  /// @param nbd
  /// @param lsave
  /// @param isave
  /// @param dsave
  /// @param lower
  /// @param upper
  void Minimize00(
      const std::function<Tv(const Matrix<Tv> &)> &function,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &gradient,
      Matrix<Tv> &startPoint, Tv *storage, Tv *work, Ti *iWORK, Ti *nbd,
      logical *lsave, Ti *isave, Tv *dsave, Matrix<Tv> *lower = nullptr,
      Matrix<Tv> *upper = nullptr);
};

/// @brief Newton optimization
class LDT_EXPORT Newton {
  Ti mK;

public:
  /// @brief Gets the required size of the storage array
  Ti StorageSize = 0;

  /// @brief Gets the required size of the work array (real)
  Ti WorkSize = 0;

  /// @brief Sets the maximum number of iterations
  Ti IterationMax = 100;

  /// @brief Sets the gradient tolerance
  Tv TolGradient = 0;

  /// @brief Sets the function tolerance
  Tv TolFunction = 1e-6;

  /// @brief Sets whether to use line search or not
  bool UseLineSearch = true;

  /// @brief Gets the number of iterations in minimize
  Ti Iteration = 0;

  /// @brief Gets the overall number of line searches
  Ti iter_line = 0;

  /// @brief Gets the value of the function in minimize
  Tv FunctionValue = NAN;

  /// @brief Gets the norm of the gradient
  Tv gnorm = NAN;

  /// @brief Absolute value of change in the value of the function in the last
  /// iteration
  Tv vf_diff = NAN;

  /// @brief Pointer to the starting value array
  Matrix<Tv> *X = nullptr;

  /// @brief Gradient in the last iteration
  Matrix<Tv> Gradient;

  /// @brief Hessian in the last iteration
  Matrix<Tv> Hessian;

  /// @brief Initializes a new instance of the class
  Newton(){};

  /// @brief Initializes a new instance of the class
  /// @param k Expected number of variables in the objective function
  Newton(Ti k);

  /// @brief Finds the minimum
  /// @param function Objective function
  /// @param gradient Gradient function
  /// @param hessian Hessian function
  /// @param x0 Starting point
  /// @param storage storage array of size \ref StorageSize
  /// @param work work array of size \ref WorkSize
  void Minimize(
      const std::function<Tv(const Matrix<Tv> &)> &function,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &gradient,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &hessian,
      Matrix<Tv> &x0, Tv *storage, Tv *work);

  /// @brief Similar to \ref Minimize with separated storages for gradient and
  /// hessian
  /// @param function
  /// @param gradient
  /// @param hessian
  /// @param x0
  /// @param storageG Storage for hessian (length = n * n)
  /// @param storageH Storage for gradient (length = n)
  /// @param work
  void Minimize2(
      const std::function<Tv(const Matrix<Tv> &)> &function,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &gradient,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &hessian,
      Matrix<Tv> &x0, Tv *storageG, Tv *storageH, Tv *work);

private:
  void minimize(
      const std::function<Tv(const Matrix<Tv> &)> &function,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &gradient,
      const std::function<void(const Matrix<Tv> &, Matrix<Tv> &)> &hessian,
      Matrix<Tv> &x0, Tv *work);
};

/*

*/

/// @brief Nelder-Mead optimization.
class LDT_EXPORT NelderMead {

public:
  /// @brief Gets the required size of the work array (real)
  Ti WorkSize = 0;

  Ti StorageSize = 0;

  /// @brief Sets the reflection coefficient (default = 1.0). This parameter
  /// controls how far the worst point is reflected through the centroid of the
  /// remaining points.
  Tv ParamReflection = 1.0;

  /// @brief Sets the expansion coefficient (default = 2.0). This parameter
  /// controls how far the reflected point is expanded along the line connecting
  /// it to the centroid.
  Tv ParamExpansion = 2.0;

  /// @brief Sets the contraction coefficient (default = 0.5). This
  /// parameter controls how far the worst point is contracted towards the
  /// centroid.
  Tv ParamContraction = 0.5;

  /// @brief Sets the shrink coefficient (default = 0.5). This parameter
  /// controls how much the simplex is shrunk towards the best point when all
  /// other moves are rejected.
  Tv ParamShrink = 0.5;

  /// @brief Sets the tolerance for convergence (default = 1e-6). The
  /// algorithm terminates when the difference between the best and worst points
  /// in the simplex is less than this value.
  Tv Tolerance = 1e-6;

  /// @brief Sets the maximum number of iterations (default = 1000). The
  /// algorithm terminates after this many iterations if it has not converged
  /// yet.
  Ti MaxIteration = 1000;

  /// @brief Gets the current number of iterations
  Ti Iter = 0;

  /// @brief difference between the best and worst points in the simplex at the
  /// last iteration.
  Tv Diff = NAN;

  /// @brief Minimum value of the objective function. The value might be added
  /// with a penalty if the problem is bounded
  Tv Min = NAN;

  /// @brief After Minimize, the optimal solution found by the
  /// algorithm
  Matrix<Tv> Result;

  /// @brief Initializes a new instance of the class
  /// @param n Expected number of variables
  NelderMead(Ti n);

  /// @brief Minimizes a given objective function
  ///
  /// @param Tv The value type of the objective function.
  /// @param objective The objective function to minimize.
  /// @param start The starting point for the optimization.
  /// @param work work array
  /// @param storage  storage array
  ///@param lower Use it to restrict the solution
  /// @param upper use it to restrict the solution
  /// @param penaltyMultiplier restricting the solution is done by defining a
  /// penalty function. You can increase the penalty by this multiplier.
  void Minimize(const std::function<Tv(const Matrix<Tv> &)> &objective,
                const Matrix<Tv> &start, Tv *work, Tv *storage,
                const Matrix<Tv> *lower = nullptr,
                const Matrix<Tv> *upper = nullptr, Tv penaltyMultiplier = 10);

  /// @brief Nelder-mead optimization for univariate function
  /// @param f univariate function
  /// @param x0 Initial guess
  /// @param step initial size of the simplex. The simplex is a geometrical
  /// figure that the algorithm uses to search for the minimum of the function.
  /// In the case of a univariate function, the simplex is just a line segment.
  /// The step argument determines the initial length of this line segment.
  /// @param max_iter maximum number of iterations that the algorithm will
  /// perform. If the algorithm does not converge to a minimum within this
  /// number of iterations, it will stop and return the current best estimate
  /// for the minimum.
  /// @param tol
  /// @param x_min Lower bound for x. See the remark for a general constraint
  /// optimization.
  /// @param x_max A constrain on maximum value of x. See the remark for a
  /// general constraint optimization.
  /// @return <Minimum X, Current Iteration>
  /// @remark For a constraint optimization, add a penalty to the objective
  /// function. e.g. if g(x) <= 0, modify the objective function to be: auto
  /// f_constrained = [f](double x) {
  ///  double P = 0.0;
  ///  if (g(x) > 0) {
  ///      P = C * std::pow(g(x), 2);
  ///  }
  /// };
  ///  return f(x) + P;
  /// Use a large but finite value for C. Infinity or relatively large number
  /// can cause numerical issues and may result in an unstable optimization
  /// process.
  static std::tuple<Tv, Ti>
  Minimize1(const std::function<Tv(const Tv &)> &objective, Tv x0,
            Tv step = 0.1, int max_iter = 100, Tv tol = 1e-6, Tv x_min = NAN,
            Tv x_max = NAN);
};

} // namespace ldt
