


#' Publicly accessible method for determining the current backend.
#'
#' @description
#'
#' # Examples
#' ```{r}
#' config_backend()
#' ```
#'
#' @returns
#' String, the name of the backend Keras is currently using. One of
#' `"tensorflow"`, `"torch"`, or `"jax"`.
#'
#' @export
#' @family config backend
#' @family backend
#' @family config
#' @seealso
#' [use_backend()]
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/backend>
#' @tether keras.config.backend
config_backend <-
function ()
{
    keras$config$backend()
}


#' Return the value of the fuzz factor used in numeric expressions.
#'
#' @description
#'
#' # Examples
#' ```{r}
#' config_epsilon()
#' ```
#'
#' @returns
#' A float.
#'
#' @export
#' @family config backend
#' @family backend
#' @family config
#' @seealso
#' + <https://keras.io/api/utils/config_utils#epsilon-function>
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/epsilon>
#' @tether keras.config.epsilon
config_epsilon <-
function ()
{
    keras$config$epsilon()
}


#' Return the default float type, as a string.
#'
#' @description
#' E.g. `'bfloat16'` `'float16'`, `'float32'`, `'float64'`.
#'
#' # Examples
#' ```{r}
#' keras3::config_floatx()
#' ```
#'
#' @returns
#' String, the current default float type.
#'
#' @export
#' @family config backend
#' @family backend
#' @family config
#' @seealso
#' + <https://keras.io/api/utils/config_utils#floatx-function>
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/floatx>
#' @tether keras.config.floatx
config_floatx <-
function ()
{
   keras$config$floatx()
}


#??
function(x) {
  # config_floatx?
  if(missing(x))
    keras$config$floatx()
  else
    keras$config$set_floatx(x)
}




#' Return the default image data format convention.
#'
#' @description
#'
#' # Examples
#' ```{r}
#' config_image_data_format()
#' ```
#'
#' @returns
#' A string, either `'channels_first'` or `'channels_last'`.
#'
#' @export
#' @family config backend
#' @family backend
#' @family config
#' @seealso
#' + <https://keras.io/api/utils/config_utils#imagedataformat-function>
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/image_data_format>
#' @tether keras.config.image_data_format
config_image_data_format <-
function ()
{
    args <- capture_args()
    do.call(keras$config$image_data_format, args)
}


#' Set the value of the fuzz factor used in numeric expressions.
#'
#' @description
#'
#' # Examples
#' ```{r}
#' config_epsilon()
#' ```
#'
#' ```{r}
#' config_set_epsilon(1e-5)
#' config_epsilon()
#' ```
#'
#' ```{r}
#' # Set it back to the default value.
#' config_set_epsilon(1e-7)
#' ```
#'
#' @param value
#' float. New value of epsilon.
#'
#' @returns No return value, called for side effects.
#' @export
#' @family config backend
#' @family backend
#' @family config
#' @seealso
#' + <https://keras.io/api/utils/config_utils#setepsilon-function>
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/set_epsilon>
#' @tether keras.config.set_epsilon
config_set_epsilon <-
function (value)
{
    args <- capture_args()
    do.call(keras$config$set_epsilon, args)
}


#' Set the default float dtype.
#'
#' @description
#'
#' # Note
#' It is not recommended to set this to `"float16"` for training,
#' as this will likely cause numeric stability issues.
#' Instead, mixed precision, which leverages
#' a mix of `float16` and `float32`. It can be configured by calling
#' `keras3::keras$mixed_precision$set_dtype_policy('mixed_float16')`.
#'
#' # Examples
#' ```{r}
#' config_floatx()
#' ```
#'
#' ```{r}
#' config_set_floatx('float64')
#' config_floatx()
#' ```
#'
#' ```{r}
#' # Set it back to float32
#' config_set_floatx('float32')
#' ```
#'
#' # Raises
#' ValueError: In case of invalid value.
#'
#' @param value
#' String; `'bfloat16'`, `'float16'`, `'float32'`, or `'float64'`.
#'
#' @returns No return value, called for side effects.
#' @export
#' @family config backend
#' @family backend
#' @family config
#' @seealso
#' + <https://keras.io/api/utils/config_utils#setfloatx-function>
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/set_floatx>
#' @tether keras.config.set_floatx
config_set_floatx <-
function (value)
{
    keras$config$set_floatx(value)
}


#' Set the value of the image data format convention.
#'
#' @description
#'
#' # Examples
#' ```{r}
#' config_image_data_format()
#' # 'channels_last'
#' ```
#'
#' ```{r}
#' keras3::config_set_image_data_format('channels_first')
#' config_image_data_format()
#' ```
#'
#' ```{r}
#' # Set it back to `'channels_last'`
#' keras3::config_set_image_data_format('channels_last')
#' ```
#'
#' @param data_format
#' string. `'channels_first'` or `'channels_last'`.
#'
#' @returns No return value, called for side effects.
#' @export
#' @family config backend
#' @family backend
#' @family config
#' @seealso
#' + <https://keras.io/api/utils/config_utils#setimagedataformat-function>
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/set_image_data_format>
#' @tether keras.config.set_image_data_format
config_set_image_data_format <-
function (data_format)
{
    args <- capture_args()
    do.call(keras$config$set_image_data_format, args)
}


#' Disables safe mode globally, allowing deserialization of lambdas.
#'
#' @returns No return value, called for side effects.
#' @export
#' @family saving
#' @family config
# @seealso
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/enable_unsafe_deserialization>
#' @tether keras.config.enable_unsafe_deserialization
config_enable_unsafe_deserialization <-
function ()
{
    args <- capture_args()
    do.call(keras$config$enable_unsafe_deserialization, args)
}


#' Turn off interactive logging.
#'
#' @description
#' When interactive logging is disabled, Keras sends logs to `absl.logging`.
#' This is the best option when using Keras in a non-interactive
#' way, such as running a training or inference job on a server.
#'
#' @returns No return value, called for side effects.
#' @export
#' @family io utils
#' @family utils
#' @family config
# @seealso
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/disable_interactive_logging>
#' @tether keras.config.disable_interactive_logging
config_disable_interactive_logging <-
function ()
{
    args <- capture_args()
    do.call(keras$config$disable_interactive_logging, args)
}


#' Turn on interactive logging.
#'
#' @description
#' When interactive logging is enabled, Keras displays logs via stdout.
#' This provides the best experience when using Keras in an interactive
#' environment such as a shell or a notebook.
#'
#' @returns No return value, called for side effects.
#' @export
#' @family io utils
#' @family utils
#' @family config
# @seealso
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/enable_interactive_logging>
#' @tether keras.config.enable_interactive_logging
config_enable_interactive_logging <-
function ()
{
    args <- capture_args()
    do.call(keras$config$enable_interactive_logging, args)
}


#' Check if interactive logging is enabled.
#'
#' @description
#' To switch between writing logs to stdout and `absl.logging`, you may use
#' [`config_enable_interactive_logging()`] and
#' [`config_disable_interactive_logging()`].
#'
#' @returns
#' Boolean, `TRUE` if interactive logging is enabled,
#' and `FALSE` otherwise.
#'
#' @export
#' @family io utils
#' @family utils
#' @family config
# @seealso
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/is_interactive_logging_enabled>
#' @tether keras.config.is_interactive_logging_enabled
config_is_interactive_logging_enabled <-
function ()
{
    args <- capture_args()
    do.call(keras$config$is_interactive_logging_enabled, args)
}


#' Turn off traceback filtering.
#'
#' @description
#' Raw Keras tracebacks (also known as stack traces)
#' involve many internal frames, which can be
#' challenging to read through, while not being actionable for end users.
#' By default, Keras filters internal frames in most exceptions that it
#' raises, to keep traceback short, readable, and focused on what's
#' actionable for you (your own code).
#'
#' See also [`config_enable_traceback_filtering()`] and
#' [`config_is_traceback_filtering_enabled()`].
#'
#' If you have previously disabled traceback filtering via
#' [`config_disable_traceback_filtering()`], you can re-enable it via
#' [`config_enable_traceback_filtering()`].
#'
#' @returns No return value, called for side effects.
#' @export
#' @family traceback utils
#' @family utils
#' @family config
# @seealso
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/disable_traceback_filtering>
#' @tether keras.config.disable_traceback_filtering
config_disable_traceback_filtering <-
function ()
{
    args <- capture_args()
    do.call(keras$config$disable_traceback_filtering, args)
}


#' Turn on traceback filtering.
#'
#' @description
#' Raw Keras tracebacks (also known as stack traces)
#' involve many internal frames, which can be
#' challenging to read through, while not being actionable for end users.
#' By default, Keras filters internal frames in most exceptions that it
#' raises, to keep traceback short, readable, and focused on what's
#' actionable for you (your own code).
#'
#' See also [`config_disable_traceback_filtering()`] and
#' [`config_is_traceback_filtering_enabled()`].
#'
#' If you have previously disabled traceback filtering via
#' [`config_disable_traceback_filtering()`], you can re-enable it via
#' [`config_enable_traceback_filtering()`].
#'
#' @returns No return value, called for side effects.
#' @export
#' @family traceback utils
#' @family utils
#' @family config
# @seealso
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/enable_traceback_filtering>
#' @tether keras.config.enable_traceback_filtering
config_enable_traceback_filtering <-
function ()
{
    args <- capture_args()
    do.call(keras$config$enable_traceback_filtering, args)
}


#' Check if traceback filtering is enabled.
#'
#' @description
#' Raw Keras tracebacks (also known as stack traces)
#' involve many internal frames, which can be
#' challenging to read through, while not being actionable for end users.
#' By default, Keras filters internal frames in most exceptions that it
#' raises, to keep traceback short, readable, and focused on what's
#' actionable for you (your own code).
#'
#' See also [`config_enable_traceback_filtering()`] and
#' [`config_disable_traceback_filtering()`].
#'
#' If you have previously disabled traceback filtering via
#' [`config_disable_traceback_filtering()`], you can re-enable it via
#' [`config_enable_traceback_filtering()`].
#'
#' @returns
#' Boolean, `TRUE` if traceback filtering is enabled,
#' and `FALSE` otherwise.
#'
#' @export
#' @family traceback utils
#' @family utils
#' @family config
# @seealso
#  + <https://www.tensorflow.org/api_docs/python/tf/keras/config/is_traceback_filtering_enabled>
#' @tether keras.config.is_traceback_filtering_enabled
config_is_traceback_filtering_enabled <-
function ()
{
    args <- capture_args()
    do.call(keras$config$is_traceback_filtering_enabled, args)
}
