% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimizers.R
\name{optimizer_ftrl}
\alias{optimizer_ftrl}
\title{Optimizer that implements the FTRL algorithm}
\usage{
optimizer_ftrl(
  learning_rate = 0.001,
  learning_rate_power = -0.5,
  initial_accumulator_value = 0.1,
  l1_regularization_strength = 0,
  l2_regularization_strength = 0,
  l2_shrinkage_regularization_strength = 0,
  beta = 0,
  weight_decay = NULL,
  clipnorm = NULL,
  clipvalue = NULL,
  global_clipnorm = NULL,
  use_ema = FALSE,
  ema_momentum = 0.99,
  ema_overwrite_frequency = NULL,
  jit_compile = TRUE,
  name = "Ftrl",
  ...
)
}
\arguments{
\item{learning_rate}{A \code{Tensor}, floating point value, a schedule that is a
\code{tf.keras.optimizers.schedules.LearningRateSchedule}, or a callable that
takes no arguments and returns the actual value to use. The learning
rate.  Defaults to 0.001.}

\item{learning_rate_power}{A float value, must be less or equal to zero.
Controls how the learning rate decreases during training. Use zero for a
fixed learning rate.}

\item{initial_accumulator_value}{The starting value for accumulators. Only zero
or positive values are allowed.}

\item{l1_regularization_strength}{A float value, must be greater than or equal
to zero. Defaults to 0.0.}

\item{l2_regularization_strength}{A float value, must be greater than or equal
to zero. Defaults to 0.0.}

\item{l2_shrinkage_regularization_strength}{A float value, must be greater than
or equal to zero. This differs from L2 above in that the L2 above is a
stabilization penalty, whereas this L2 shrinkage is a magnitude penalty.
When input is sparse shrinkage will only happen on the active weights.}

\item{beta}{A float value, representing the beta value from the paper. Defaults
to 0.0.}

\item{weight_decay}{Float, defaults to NULL. If set, weight decay is applied.}

\item{clipnorm}{Float. If set, the gradient of each weight is individually
clipped so that its norm is no higher than this value.}

\item{clipvalue}{Float. If set, the gradient of each weight is clipped to be no
higher than this value.}

\item{global_clipnorm}{Float. If set, the gradient of all weights is clipped so
that their global norm is no higher than this value.}

\item{use_ema}{Boolean, defaults to FALSE. If TRUE, exponential moving average
(EMA) is applied. EMA consists of computing an exponential moving
average of the weights of the model (as the weight values change after
each training batch), and periodically overwriting the weights with
their moving average.}

\item{ema_momentum}{Float, defaults to 0.99. Only used if \code{use_ema=TRUE}. This is  # noqa: E501
the momentum to use when computing the EMA of the model's weights:
\code{new_average = ema_momentum * old_average + (1 - ema_momentum) * current_variable_value}.}

\item{ema_overwrite_frequency}{Int or NULL, defaults to NULL. Only used if
\code{use_ema=TRUE}. Every \code{ema_overwrite_frequency} steps of iterations, we
overwrite the model variable by its moving average. If NULL, the optimizer  # noqa: E501
does not overwrite model variables in the middle of training, and you
need to explicitly overwrite the variables at the end of training
by calling \code{optimizer.finalize_variable_values()} (which updates the model  # noqa: E501
variables in-place). When using the built-in \code{fit()} training loop, this
happens automatically after the last epoch, and you don't need to do
anything.}

\item{jit_compile}{Boolean, defaults to TRUE. If TRUE, the optimizer will use XLA  # noqa: E501
compilation. If no GPU device is found, this flag will be ignored.}

\item{name}{String. The name to use
for momentum accumulator weights created by
the optimizer.}

\item{...}{Used for backward and forward compatibility}
}
\value{
Optimizer for use with \code{\link{compile.keras.engine.training.Model}}.
}
\description{
Optimizer that implements the FTRL algorithm
}
\details{
"Follow The Regularized Leader" (FTRL) is an optimization algorithm
developed at Google for click-through rate prediction in the early 2010s. It
is most suitable for shallow models with large and sparse feature spaces.
The algorithm is described by
\href{https://research.google.com/pubs/archive/41159.pdf}{McMahan et al., 2013}.
The Keras version has support for both online L2 regularization
(the L2 regularization described in the paper
above) and shrinkage-type L2 regularization
(which is the addition of an L2 penalty to the loss function).

Initialization:

\if{html}{\out{<div class="sourceCode python">}}\preformatted{n = 0
sigma = 0
z = 0
}\if{html}{\out{</div>}}

Update rule for one variable \code{w}:

\if{html}{\out{<div class="sourceCode python">}}\preformatted{prev_n = n
n = n + g ** 2
sigma = (n ** -lr_power - prev_n ** -lr_power) / lr
z = z + g - sigma * w
if abs(z) < lambda_1:
  w = 0
else:
  w = (sgn(z) * lambda_1 - z) / ((beta + sqrt(n)) / alpha + lambda_2)
}\if{html}{\out{</div>}}

Notation:
\itemize{
\item \code{lr} is the learning rate
\item \code{g} is the gradient for the variable
\item \code{lambda_1} is the L1 regularization strength
\item \code{lambda_2} is the L2 regularization strength
\item \code{lr_power} is the power to scale n.
}

Check the documentation for the \code{l2_shrinkage_regularization_strength}
parameter for more details when shrinkage is enabled, in which case gradient
is replaced with a gradient with shrinkage.
}
\seealso{
\itemize{
\item \url{https://www.tensorflow.org/api_docs/python/tf/keras/optimizers/experimental/Ftrl}
}

Other optimizers: 
\code{\link{optimizer_adadelta}()},
\code{\link{optimizer_adagrad}()},
\code{\link{optimizer_adamax}()},
\code{\link{optimizer_adam}()},
\code{\link{optimizer_nadam}()},
\code{\link{optimizer_rmsprop}()},
\code{\link{optimizer_sgd}()}
}
\concept{optimizers}
