% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynLong.R
\name{dynLong}
\alias{dynLong}
\title{Dynamic predictions for the longitudinal data sub-model}
\usage{
dynLong(object, newdata, newSurvData = NULL, u = NULL,
  type = "first-order", M = 200, scale = 1.6, ci, progress = TRUE,
  ntimes = 100, level = 1)
}
\arguments{
\item{object}{an object inheriting from class \code{mjoint} for a joint model
of time-to-event and multivariate longitudinal data.}

\item{newdata}{a list of \code{data.frame} objects for each longitudinal
outcome for a single new patient in which to interpret the variables named
in the \code{formLongFixed} and \code{formLongRandom} formulae of
\code{object}. As per \code{\link{mjoint}}, the \code{list} structure
enables one to include multiple longitudinal outcomes with different
measurement protocols. If the multiple longitudinal outcomes are measured
at the same time points for each patient, then a \code{data.frame} object
can be given instead of a \code{list}. It is assumed that each data frame
is in long format.}

\item{newSurvData}{a \code{data.frame} in which to interpret the variables
named in the \code{formSurv} formulae from the \code{mjoint} object. This
is optional, and if omitted, the data will be searched for in
\code{newdata}. Note that no event time or censoring indicator data are
required for dynamic prediction. Defaults to \code{newSurvData=NULL}.}

\item{u}{an optional time that must be greater than the last observed
measurement time. If omitted (default is \code{u=NULL}), then conditional
failure probabilities are reported for all observed failure times in the
\code{mjoint} object data.}

\item{type}{a character string for whether a first-order
(\code{type="first-order"}) or Monte Carlo simulation approach
(\code{type="simulated"}) should be used for the dynamic prediction.
Defaults to the computationally faster first-order prediction method.}

\item{M}{for \code{type="simulated"}, the number of simulations to performs.
Default is \code{M=200}.}

\item{scale}{a numeric scalar that scales the variance parameter of the
proposal distribution for the Metropolis-Hastings algorithm, which
therefore controls the acceptance rate of the sampling algorithm.}

\item{ci}{a numeric value with value in the interval \eqn{(0, 1)} specifying
the confidence interval level for predictions of \code{type='simulated'}.
If missing, defaults to \code{ci=0.95} for a 95\% confidence interval. If
\code{type='first-order'} is used, then this argument is ignored.}

\item{progress}{logical: should a progress bar be shown on the console to
indicate the percentage of simulations completed? Default is
\code{progress=TRUE}.}

\item{ntimes}{an integer controlling the number of points to discretize the
extrapolated time region into. Default is \code{ntimes=100}.}

\item{level}{an optional integer giving the level of grouping to be used in
extracting the residuals from object. Level values increase from outermost
to innermost grouping, with level 0 corresponding to the population model
fit and level 1 corresponding to subject-specific model fit. Defaults to
\code{level=1}.}
}
\value{
A list object inheriting from class \code{dynLong}. The list returns
  the arguments of the function and a list containing \emph{K}
  \code{data.frame}s of 2 columns, with first column (named
  \code{timeVar[k]}; see \code{\link{mjoint}}) denoting times and the second
  column (named \code{y.pred}) denoting the expected outcome at each time
  point.
}
\description{
Calculates the conditional expected longitudinal values for a
  \emph{new} subject from the last observation time given their longitudinal
  history data and a fitted \code{mjoint} object.
}
\details{
Dynamic predictions for the longitudinal data sub-model based on an
  observed measurement history for the longitudinal outcomes of a new subject
  are based on either a first-order approximation or Monte Carlo simulation
  approach, both of which are described in Rizopoulos (2011). Namely, given
  that the subject was last observed at time \emph{t}, we calculate the
  conditional expectation of each longitudinal outcome at time \emph{u} as

  \deqn{E[y_k(u) | T \ge t, y, \theta] \approx x^T(u)\beta_k +
  z^T(u)\hat{b}_k,}

  where \eqn{T} is the failure time for the new subject, and \eqn{y} is the
  stacked-vector of longitudinal measurements up to time \emph{t}.

  \strong{First order predictions}

  For \code{type="first-order"}, \eqn{\hat{b}} is the mode of the posterior
  distribution of the random effects given by

  \deqn{\hat{b} = {\arg \max}_b f(b | y, T \ge t; \theta).}

  The predictions are based on plugging in \eqn{\theta = \hat{\theta}}, which
  is extracted from the \code{mjoint} object.

  \strong{Monte Carlo simulation predictions}

  For \code{type="simulated"}, \eqn{\theta} is drawn from a multivariate
  normal distribution with means \eqn{\hat{\theta}} and variance-covariance
  matrix both extracted from the fitted \code{mjoint} object via the
  \code{coef()} and \code{vcov()} functions. \eqn{\hat{b}} is drawn from the
  the posterior distribution of the random effects

  \deqn{f(b | y, T \ge t; \theta)}

  by means of a Metropolis-Hasting algorithm with independent multivariate
  non-central \emph{t}-distribution proposal distributions with
  non-centrality parameter \eqn{\hat{b}} from the first-order prediction and
  variance-covariance matrix equal to \code{scale} \eqn{\times} the inverse
  of the negative Hessian of the posterior distribution. The choice os
  \code{scale} can be used to tune the acceptance rate of the
  Metropolis-Hastings sampler. This simulation algorithm is iterated \code{M}
  times, at each time calculating the conditional survival probability.
}
\examples{
\dontrun{
# Fit a joint model with bivariate longitudinal outcomes

data(heart.valve)
hvd <- heart.valve[!is.na(heart.valve$log.grad) & !is.na(heart.valve$log.lvmi), ]

fit2 <- mjoint(
    formLongFixed = list("grad" = log.grad ~ time + sex + hs,
                         "lvmi" = log.lvmi ~ time + sex),
    formLongRandom = list("grad" = ~ 1 | num,
                          "lvmi" = ~ time | num),
    formSurv = Surv(fuyrs, status) ~ age,
    data = list(hvd, hvd),
    inits = list("gamma" = c(0.11, 1.51, 0.80)),
    timeVar = "time",
    verbose = TRUE)

hvd2 <- droplevels(hvd[hvd$num == 1, ])
dynLong(fit2, hvd2)
dynLong(fit2, hvd2, u = 7) # outcomes at 7-years only

out <- dynLong(fit2, hvd2, type = "simulated")
out
}
}
\references{
Rizopoulos D. Dynamic predictions and prospective accuracy in joint models
for longitudinal and time-to-event data. \emph{Biometrics}. 2011;
\strong{67}: 819–829.
}
\seealso{
\code{\link{mjoint}}, \code{\link{dynSurv}}.
}
\author{
Graeme L. Hickey (\email{graeme.hickey@liverpool.ac.uk})
}
\keyword{multivariate}
