#' Create data to run IRT model
#' 
#' To run an IRT model using \code{idealstan}, you must first process your data using the \code{id_make} function. 
#' 
#' @param score_data Your person-item (legislator-bill) matrix in which persons (legislators) are in rows and 
#'    items (bills) are in columns. The cells of the matrix should contain consecutive integers \eqn{1,...K} for 
#'    ordinal \eqn{K} outcomes or integers \eqn{[0,1]} for a binary outcome in which 0 equals no (or incorrect) and 
#'    1 equals yes (or correct). If absences/missing data are included, they should be coded as the highest category \eqn{K+1} for
#'    ordinal outcomes and 2 for binary outcomes.
#' @param simul_data Optionally, data that has been generated by the \code{\link{id_sim_gen}} function.
#' @param person_cov A matrix of person (legislator) covariates for hierarchical modeling. If multiple time points are used, then should be a 
#'  multi-column matrix with one column for each time point.
#' @param item_cov A one-column matrix of item (bill) covariates for hierarchical modeling of item (bill) discrimination parameters in the non-inflated model, i.e., non-missing model
#' @param item_cov_miss A one-column matrix of item (bill) covariates for hierarchical modeling of item (bill) discrimination parameters in the inflated model (missing data model)
#' @param person_data An optional data frame of information about the persons (legislators). Should include a \code{person.names} column containing names of
#'  persons (legislators) and optionally a \code{group} column with names of any groupings of the persons (legislators), such as parties or blocs (used for visualization).
#' @param item_data An optional data frame of item/bill labels and other information used for visualization. Should include \code{item.names} column 
#'  containing item/bill labels; other columns are optional.
#' @param miss_val The value (numeric or character) that indicate missing data/absences in the data. If there are multiple possible values, 
#'  pass along a numeric or character vector of all such values. If missing data is coded as \code{NA}, simply leave this parameter at the default, \code{NA}.
#' @param high_val The value (numeric or character) that indicate the highest ordinal outcome possible, such as yes in a vote dataset or correct in a test examination.
#'  If there are multiple possible values, 
#'  pass along a numeric or character vector of all such values.
#' @param low_val The value (numeric or character) that indicates the lowest ordinal outcome possible, such as no votes in a vote dataset or incorrect in a test examination.
#'  If there are multiple possible values, 
#'  pass along a numeric or character vector of all such values.
#' @param middle_val The value (numeric or character) that indicate values between the lowest and highest categories, such as abstention in voting data or "Neither Agree nor Disagree" in likert scales.
#'  If there are multiple possible values, 
#'  pass along a numeric or character vector of all such values in correct order (lower to higher values).
#'  If there are no middle values (binary outcome), set to \code{NULL}.
#' @param ordinal Whether or not the data contain ordinal responses. If \code{TRUE}, middle values/abstentions are used as a middle category in constructing the outcome.
#'  Otherwise the response is assumed to be binary (yes/no) or (correct/incorrect).
#' @param time An optional integer vector of length equal to the number of columns in the response matrix (i.e., the number of items or bills)
#'  that indicates for each item/bill which time point that item/bill belongs to. This time vector is only used for models in which person/legislator
#'  ideal points are allowed to vary over time. 
#' @param outcome_label_type Whether to use pre-set labels for the outcome values. If set to \code{bills}, the default, will label the outcome as \code{c('No','Abstain','Yes','Absent')} if there are 3 possible categories.
#'  Otherwise, either pass \code{NULL} to this option to use integers for labels or a character vector equal to the number of categories in the
#'  outcome. Used for visualization.
#' @param exclude_level A vector of any values that should be treated as \code{NA} in the response matrix. Unlike the \code{middle_val} parameter, these values will be dropped from the data before estimation rather than modeled explicitly.
#' @param inflate If \code{TRUE}, the score matrix is set up to enable modeling of missing data/absences (\code{miss_val}) as an inflation model in \code{\link{id_estimate}}
#' @param simulation If \code{TRUE}, simulated values are saved in the \code{idealdata} object for later plotting with the \code{\link{id_plot_sims}} function
#' @param include_pres If \code{FALSE} and \code{score_data} is a \code{rollcall} object, drop the first row which often represents tiebreaker votes cast by the Vice President in the US Senate.
#' @return A \code{idealdata} object that can then be used in the \code{\link{id_estimate}} function to fit a model.
#' @export
#' @import rstan
#' @import dplyr
#' @importFrom tidyr gather spread
#' @import bayesplot
#' @import rstantools
#' @import Rcpp
#' @import methods
#' @importFrom stats dbinom median plogis quantile reorder rexp rlnorm runif sd step rnorm
#' @useDynLib idealstan, .registration = TRUE
#' @examples 
#' # You can either use a pscl rollcall object or a vote/score matrix 
#' # where persons/legislators are in the rows
#' # and items/bills are in the columns
#' 
#' library(dplyr)
#' 
#' # First, using a rollcall object with the 114th Senate's rollcall votes:
#' 
#' data('senate114')
#' 
#' to_idealstan <-   id_make(score_data = senate114,
#' ordinal = FALSE,
#' include_pres=FALSE)
#' 
id_make <- function(score_data=NULL,simul_data=NULL,
                           person_cov=NULL,item_cov=NULL,
                           item_cov_miss=NULL,
                           person_data=NULL,item_data=NULL,
                           miss_val=NA,high_val=3L,low_val=1L,middle_val=2L,
                           ordinal=TRUE,time=NULL,
                           outcome_label_type='votes',
                           exclude_level=NULL,inflate=TRUE,simulation=FALSE,
                    include_pres=FALSE) {
  
  
  if(class(score_data)=='rollcall') {
    miss_val <- 9
    low_val <- 6
    high_val <- 1
    exclude_level <- c(3,7)
    row_names <- row.names(score_data$legis.data)
    if(include_pres==F) {
      person_data <- slice(score_data$legis.data,-1)
      row_names <- row_names[-1]
    }
    
    person_data <-  mutate(person_data,group=party,person.names=row_names)
    
    if(include_pres==F) {
      score_data <- score_data$votes[-1,]
    } else {
      score_data <- score_data$votes
    }
    
    row.names(score_data) <- row_names
  }
  
  
  # recode missing into something that works
  if(is.na(miss_val) && inflate==TRUE) {
    if(sum(is.na(score_data))==0) stop('To use missing-data model, you must have at least one missing value in the score matrix.')
    suppressWarnings(score_data <- apply(score_data,2,function(c) {
      if(is.numeric(c)) {
          max_c <- max(c,na.rm=T)
          c[is.na(c)] <- max_c +1
      } else {
        c[is.na(c)] <- 'Missing'
        }
      return(c)
      
    }))
    if(is.numeric(score_data[,1])) {
      miss_val <- max(score_data)
    } else {
      miss_val <- 'Missing'
    }
  }

  if(ordinal==TRUE & inflate==TRUE) {
    votes <- c(low_val,middle_val,high_val,miss_val)
    vote_int <- as.integer(factor(votes,levels=votes))
    names(vote_int) <- votes
    vote_labels <-  c('No','Abstain','Yes','Absent')
    miss_val <- vote_int[length(vote_int)]
  } else if(ordinal==FALSE & inflate==TRUE) {
    votes <- c(low_val,high_val,miss_val)
    vote_int <- as.integer(factor(votes,levels=votes))
    names(vote_int) <- votes
    vote_labels <-  c('No','Yes','Absent')
    miss_val <- vote_int[length(vote_int)]
  } else if(ordinal==TRUE & inflate==FALSE)  {
    votes <- c(low_val,middle_val,high_val)
    vote_int <- as.integer(factor(votes,levels=votes))
    names(vote_int) <- votes
    vote_labels <-  c('No','Abstain','Yes')
    miss_val <- NA
  } else {
    votes <- c(low_val,high_val)
    vote_int <- as.integer(factor(votes,levels=votes))
    names(vote_int) <- votes
    vote_labels <-  c('No','Yes')
    miss_val <- NA
  }
  
  if(class(score_data)=='matrix') {
    # Register all possible votes as integers, then before running the model we can change them if need be.
    cleaned <- score_data %>% apply(2,function(c) {
      as.integer(factor(c,levels=votes))
    }) 
  } else if(class(score_data)=='rollcall') {
    
  } else {
    stop('Please provide either a matrix or a rollcall object as the score_data argument.')
  }
  if(!is.null(time)) {
    if(length(time)!=ncol(cleaned)) {
      stop('Time vector must be same length as the number of columns in the vote matrix.')
    }
    time <- as.numeric(factor(time))
    max_t <- max(time)
  } else {
    time <- rep(1,ncol(cleaned))
  }
  colnames(cleaned) <- as.character(1:ncol(cleaned))
  #before doing this, need to ensure that 1) all legislators in person_data have votes and 
  #2) the rows were correctly ordered to match score_data <-> person_data
  if(!is.null(person_cov)) {
    if('data.frame' %in% class(person_cov)) {
      person_cov <- as.matrix(person_cov)
      
    } else if('matrix' %in% class(person_cov)) {
      person_cov <- array(person_cov,dim(c(ncol(person_cov),nrow(person_cov),max_t)))
    }

  } else {
    person_cov <- array(c(rep(1,nrow(person_data)),1,1),dim=c(1,nrow(person_data),1))
  }
  
  if(!is.null(item_cov)) {
    if(nrow(bill_cov)!=ncol(cleaned)) {
      stop('Item/bill covariate data must be same length as the number of columns of score/vote matrix.')
    }
  } else {
    item_cov <- matrix(rep(1,ncol(cleaned)),nrow=ncol(cleaned),ncol=1)
  }
  if(!is.null(item_cov_miss)) {
    if(nrow(item_cov_miss)!=ncol(cleaned)) {
      stop('Item/bill covariate data must be same length as the number of columns of score/vote matrix.')
    }
  } else {
    item_cov_miss <- matrix(rep(1,ncol(cleaned)),nrow=ncol(cleaned),ncol=1)
  }
  
  person_data$person.names <- row.names(score_data)
  
  row.names(cleaned) <- as.character(1:nrow(cleaned))
  
  if("group" %in% names(person_data)) person_data$group <- rep('O',nrow(score_data))
  
  # check what kind of vote labels to use
  
  if(outcome_label_type=='none') {
    vote_labels <- as.character(vote_int)
  } else if(outcome_label_type!='votes') {
    vote_labels <- outcome_label_type
  }

  outobj <- new('idealdata',
      score_matrix=cleaned,
      person_data=person_data,
      person_cov=person_cov,
      time=time,
      item_cov=item_cov,
      item_cov_miss=item_cov_miss,
      vote_labels=vote_labels,
      vote_int=vote_int,
      vote_count=length(votes) - length(exclude_level),
      miss_val=miss_val)
  
  if(simulation==TRUE) {
    outobj@simul_data <- simul_data
    outobj@simulation <- simulation
  }
  return(outobj)
}

#' Estimate an \code{idealstan} model
#' 
#' This function will take a pre-processed \code{idealdata} vote/score matrix and run one of the available IRT models on the data using
#' Stan's MCMC engine.
#' 
#' To run an IRT ideal point model, you must first pre-process your data using the \code{\link{id_make}} function. Be sure to specify the correct options for the
#' kind of model you are going to run: if you want to run an ordinal and/or an inflated model, the data needs to be processed differently.
#' As of this version of \code{idealstan}, the following model types are available:
#' \enumerate{
#'   \item IRT 2-PL (binary response) ideal point model, no absence inflation
#'   \item IRT 2-PL model (binary response) with absence inflation
#'   \item Ordinal IRT (rating scale) model incorporating abstentions as middle category, no absence inflation
#'   \item Ordinal IRT (rating scale) model incorporating abstentions as middle category with absence inflation
#' }
#' Additional models are available but have not yet been tested. You can find them by searching the included stan files for more info.
#' @section Identification:
#' Identifying IRT models is challenging, and ideal point models are still more challenging because the discrimination parameters are not constrained.
#' As a result, more care must be taken to obtain estimates that are the same regardless of starting values. 
#' The parameter \code{fixtype} enables you to change the type of identification used. The default, 'vb', does not require any further
#' information from you in order for the model to be fit. In this version of identification, an unidentified model is run using
#' variational Bayesian inference (see \code{\link[rstan]{vb}}). If \code{restrict_type} is set to 'constrain_oneway', then the \code{nfix}
#' highest legislators/persons (if \code{restrict_params} is 'person') or bills/items (if \code{restrict_params} equals 'items') 
#' are used to constrain and identify the model. If \code{restrict_type} is set to 'constrain_twoway', then \code{nfix} highest and lowest legislators/persons
#' or bills/items are used to constrain and identify the model.
#' In addition, if \code{fixtype='vb'} is used, \code{auto_id} can be set to \code{TRUE}. This will run additional variational Bayesian models
#' using the identification achieved and will see if the signs of the estimated parameters are at least 90 percent in the same direction. If so, 
#' the model is considered identified. If not, the function will re-run and will increase \code{nfix} by one to see if that will identify the model, 
#' ad infinitum.
#' If \code{fixtype} is set to 'constrained', then identification is achieved by constraining a specified list of legislators/persons or
#' bills/items. If \code{restrict_type} is 'constrain_oneway', then the indices of all constrained legislators/persons or bills/items
#' should be included as the row or column indices of these parameters in the response matrix as a vector in \code{restrict_ind_high}.
#' If \code{restrict_type} is 'constrain_twoway,' then the indices of high constrained parameters should go in 
#' \code{restrict_ind_high} and the indices of low constrained parameters in \code{restrict_ind_low}. The numbers of high and low
#' constrained parameters should be equal. To pick the parameter to constrain, set \code{restrict_params} to 'person' for legislators/persons,
#' 'discrim_miss' for absence-inflated bill discrimination parameters, and 'discrim_reg' for non-inflated bill discrimination parameters.
#' If \code{fixtype} is set to 'pinned', then identification is achieved via pinning with very tight priors a set of legislators/persons or bills/items.
#' The indices of legislators/persons (i.e. row indices in the response matrix) or bills/items (column indices in the response matrix) should be passed as a vector
#' to \code{restrict_ind_high}, while \code{restrict_ind_low} should be left blank. The particular values to pin these parameters is passed as a 
#' numeric vector to \code{pin_vals}.
#' @param idealdata An object produced by the \code{\link{id_make}} containing a score/vote matrix for use for estimation & plotting
#' @param model_type An integer reflecting the kind of model to be estimated. See below.
#' @param use_subset Whether a subset of the legislators/persons should be used instead of the full response matrix
#' @param sample_it Whether or not to use a random subsample of the response matrix. Useful for testing.
#' @param subset_group If person/legislative data was included in the \code{\link{id_make}} function, then you can subset by
#' any value in the \code{$group} column of that data if \code{use_subset} is \code{TRUE}.
#' @param subset_person A list of character values of names of persons/legislators to use to subset if \code{use_subset} is 
#' \code{TRUE} and person/legislative data was included in the \code{\link{id_make}} function with the required \code{$person.names}
#' column
#' @param sample_size If \code{sample_it} is \code{TRUE}, this value reflects how many legislators/persons will be sampled from
#' the response matrix
#' @param nchains The number of chains to use in Stan's sampler. Minimum is one. See \code{\link[rstan]{stan}} for more info.
#' @param niters The number of iterations to run Stan's sampler. Shouldn't be set much lower than 500. See \code{\link[rstan]{stan}} for more info.
#' @param use_vb Whether or not to use Stan's variational Bayesian inference engine instead of full Bayesian inference. Pros: it's much faster.
#' Cons: it's not quite as accurate. See \code{\link[rstan]{vb}} for more info.
#' @param nfix An integer specifying the number of parameters to constrain (for both high and low) if \code{fixtype} is set to \code{'vb'}
#' @param warmup The number of iterations to use to calibrate Stan's sampler on a given model. Shouldn't be less than 100. 
#' See \code{\link[rstan]{stan}} for more info.
#' @param ncores The number of cores in your computer to use for parallel processing in the Stan engine. 
#' See \code{\link[rstan]{stan}} for more info.
#' @param fixtype Sets the particular kind of identification used on the model, could be one of 'vb','constrained' or 'pinned'.
#'  See details for more information.
#' @param restrict_type Determines whether constraints will only be set on one direction 'constrain_oneway' or both high/low
#' constrains 'constrain_twoway'. Ignored if \code{fixtype} is equal to 'pinned'.
#' @param restrict_params Determines which parameters in the model are used for identification. If \code{fixtype} is 'vb', can be
#' either 'person' (person/legislator ideal points) or 'items' (discrimination parameters for items/bills). Otherwise, the value must be one
#' of the available classes of parameters: 'person' for persons/legislators, 'discrim_miss' for the missing-inflated item/bill discrimination
#' parameters, or 'discrim_reg' for the non-inflated item/bill discrimination parameters.
#' @param auto_id If \code{TRUE}, a variational Bayesian version of your model will be run using \code{\link[rstan]{vb}} in a 
#' non-identified form. Depending on what is set via \code{restrict_type} and \code{restrict_params}, bills/items or 
#' persons/legislators will be used to constrain and identify the model. 
#' @param pin_vals If \code{fixtype} is set to 'pinned', then a vector of numeric values of which to pin the legislators/persons or 
#' bills/items should be given.
#' @param restrict_ind_high If \code{fixtype} is not "vb", the particular indices of legislators/persons or bills/items to constrain high
#' @param restrict_ind_low If \code{fixtype} is not "vb", the particular indices of legislators/persons or bills/items to constrain low. 
#' (Note: not used if values are pinned).
#' @param discrim_reg_sd Set the prior standard deviation of the bimodal prior for the discrimination parameters for the non-inflated model.
#' @param discrim_miss_sd Set the prior standard deviation of the bimodal prior for the discrimination parameters for the inflated model.
#' @param person_sd Set the prior standard deviation for the legislators (persons) parameters
#' @param diff_reg_sd Set the prior standard deviation for the bill (item) intercepts for the non-inflated model.
#' @param diff_miss_sd Set the prior standard deviation for the bill (item) intercepts for the inflated model.
#' @param restrict_sd Set the prior standard deviation for constrained parameters
#' @param restrict_low_bar Set the constraint threshold for constrained parameters (parameter must be lower than this bar and no greater than zero)
#' @param restrict_high_bar Set the constraint threshold for constrained parameters (parameter must be higher than this bar and no less than zero)
#' @param restrict_alpha This is the scale (alpha) parameter passed to the gamma prior if exactly two item/person parameters are constrained, each high or low. The gamma prior pushes these two
#' polar parameters apart. A higher value will push these two poles farther apart, which will help identification. 
#' @param restrict_beta The beta (shape) parameter passed to the gamma prior if exactly two item/person parameters are constrained, each high or low. The gamma prior pushes these two
#' polar parameters apart.
#' @param ... Additional parameters passed on to Stan's sampling engine. See \code{\link[rstan]{stan}} for more information.
#' @return A fitted \code{\link{idealstan}} object that contains posterior samples of all parameters either via full Bayesian inference
#' or a variational approximation if \code{use_vb} is set to \code{TRUE}. This object can then be passed to the plotting functions for further analysis.
#' @seealso \code{\link{id_make}} for pre-processing data,
#' \code{\link{id_plot_legis}} for plotting results,
#' \code{\link{summary}} for obtaining posterior quantiles,
#' \code{\link{posterior_predict}} for producing predictive replications.
#' @examples
#' # First we can simulate data for an IRT 2-PL model that is inflated for missing data
#' library(ggplot2)
#' library(dplyr)
#' 
#' # This code will take at least a few minutes to run 
#' \dontrun{
#' bin_irt_2pl_abs_sim <- id_sim_gen(ordinal=FALSE,
#'                                   absence=TRUE,
#'                                   absence_diff_mean=0)
#' 
#' # Now we can put that directly into the id_estimate function 
#' # to get full Bayesian posterior estimates
#' # We will constrain discrimination parameters 
#' # for identification purposes based on the true simulated values
#' 
#' bin_irt_2pl_abs_est <- id_estimate(bin_irt_2pl_abs_sim,
#'                        model_type=2,
#'                        restrict_ind_high = 
#'                        sort(bin_irt_2pl_abs_sim@simul_data$true_reg_discrim,
#'                        decreasing=TRUE,
#'                        index=TRUE)$ix[1:3],
#'                        restrict_ind_low = 
#'                        sort(bin_irt_2pl_abs_sim@simul_data$true_reg_discrim,
#'                        decreasing=FALSE,
#'                        index=TRUE)$ix[1:3],
#'                        restrict_params = 'discrim_reg', 
#'                        restrict_type = 'constrain_twoway',
#'                        fixtype='constrained',
#'                        ncores=2,
#'                        nchains=2)
#'                                    
#' # We can now see how well the model recovered the true parameters
#' 
#' id_sim_coverage(bin_irt_2pl_abs_est) %>% 
#'          bind_rows(.id='Parameter') %>% 
#'          ggplot(aes(y=avg,x=Parameter)) +
#'            stat_summary(fun.args=list(mult=1.96)) + 
#'            theme_minimal()
#'  }
#' 
#' # In most cases, we will use pre-existing data 
#' # and we will need to use the id_make function first
#' # We will use the full rollcall voting data 
#' # from the 114th Senate as a rollcall object
#' 
#' data('senate114')
#' 
#' # Running this model will take at least a few minutes, even with 
#' # variational inference (use_vb=T) turned on
#' \dontrun{
#' 
#' to_idealstan <-   id_make(score_data = senate114,
#'                           ordinal = FALSE,
#'                           include_pres=FALSE)
#' 
#' sen_est <- id_estimate(senate_data,
#' model_type = 2,
#' use_vb = TRUE,
#' restrict_type='constrain_oneway',
#' restrict_params='person',
#' restrict_ind_high = which(row.names(senate114$votes[-1,])=='SASSE (R NE)'),
#' auto_id=FALSE,
#' fixtype='constrained')
#' 
#' # After running the model, we can plot 
#' # the results of the person/legislator ideal points
#' 
#' id_plot_legis(sen_est)
#' }
#' 
#' @references \enumerate{
#'    \item Clinton, J., Jackman, S., & Rivers, D. (2004). The Statistical Analysis of Roll Call Data. \emph{The American Political Science Review}, 98(2), 355-370. doi:10.1017/S0003055404001194
#'    \item Bafumi, J., Gelman, A., Park, D., & Kaplan, N. (2005). Practical Issues in Implementing and Understanding Bayesian Ideal Point Estimation. \emph{Political Analysis}, 13(2), 171-187. doi:10.1093/pan/mpi010
#' }
#' @export
id_estimate <- function(idealdata=NULL,model_type=2,use_subset=FALSE,sample_it=FALSE,
                           subset_group=NULL,subset_person=NULL,sample_size=20,
                           nchains=4,niters=2000,use_vb=FALSE,nfix=3,restrict_params='person',
                           pin_vals=NULL,restrict_ind_high=NULL,
                           restrict_ind_low=NULL,
                           restrict_type='constrain_twoway',
                           fixtype='vb',warmup=floor(niters/2),ncores=4,
                           auto_id=FALSE,
                           discrim_reg_sd=3,
                           discrim_miss_sd=3,
                           person_sd=1,
                           diff_reg_sd=4,
                           diff_miss_sd=4,
                           restrict_sd=4,
                           restrict_low_bar=0,
                        restrict_high_bar=0,
                        restrict_alpha=4,
                        restrict_beta=1,
                           ...) {
  
  
  if(use_subset==TRUE || sample_it==TRUE) {
    idealdata <- subset_ideal(idealdata,use_subset=use_subset,sample_it=sample_it,subset_group=subset_group,
                              subset_person=subset_person,sample_size=sample_size)
  }
  

    hier_type <- suppressWarnings(.get_hier_type(idealdata))
    idealdata@stanmodel <- stanmodels[['irt_standard']]
   
    #Using an un-identified model with variational inference, find those parameters that would be most useful for
    #constraining/pinning to have an identified model for full Bayesian inference
  
  num_legis <- nrow(idealdata@score_matrix)
  num_bills <- ncol(idealdata@score_matrix)
  legispoints <- rep(1:num_legis,times=num_bills)
  billpoints <- rep(1:num_bills,each=num_legis)
  timepoints <- idealdata@time[billpoints]
  avg_particip <- rep(1,num_legis)
  Y <- c(idealdata@score_matrix)
  
  # check to see if we need to recode missing values from the data if the model_type doesn't handle missing data
  if(model_type %in% c(1,3) & !is.null(idealdata@miss_val)) {
    Y <- na_if(Y,idealdata@miss_val)
  }

  #Remove NA values, which should have been coded correctly in the make_idealdata function
  
    remove_nas <- !is.na(Y)
    Y <- Y[remove_nas]
    legispoints <- legispoints[remove_nas]
    billpoints <- billpoints[remove_nas]
    timepoints <- timepoints[remove_nas]

  this_data <- list(N=length(Y),
                    T=max(idealdata@time),
                    Y=Y,
                    hier_type=hier_type,
                    num_legis=num_legis,
                    num_bills=num_bills,
                    ll=legispoints,
                    bb=billpoints,
                    time=timepoints,
                    LX=dim(idealdata@person_cov)[1],
                    SRX=ncol(idealdata@item_cov),
                    SAX=ncol(idealdata@item_cov_miss),
                    legis_pred=idealdata@person_cov,
                    srx_pred=idealdata@item_cov,
                    sax_pred=idealdata@item_cov_miss,
                    particip=avg_particip,
                    model_type=model_type,
                    discrim_reg_sd=discrim_reg_sd,
                    discrim_abs_sd=discrim_miss_sd,
                    legis_sd=person_sd,
                    diff_reg_sd=diff_reg_sd,
                    diff_abs_sd=diff_miss_sd,
                    restrict_sd=restrict_sd,
                    restrict_low_bar=restrict_low_bar,
                    restrict_high_bar=restrict_high_bar,
                    restrict_alpha=restrict_alpha,
                    restrict_beta=restrict_beta)
  
  idealdata <- id_model(object=idealdata,fixtype=fixtype,model_type=model_type,this_data=this_data,
                        nfix=nfix,restrict_params=restrict_params,restrict_ind_high=restrict_ind_high,
                        restrict_ind_low=restrict_ind_low,
                        restrict_type=restrict_type,
                        auto_id=auto_id,
                        ncores=ncores)

  # Now remake the data to reflect the constrained parameters
 
  num_legis <- nrow(idealdata@score_matrix)
  num_bills <- ncol(idealdata@score_matrix)
  legispoints <- rep(1:num_legis,times=num_bills)
  billpoints <- rep(1:num_bills,each=num_legis)
  timepoints <- idealdata@time[billpoints]
  
  Y <- c(idealdata@score_matrix)
  
  # check to see if we need to recode missing values from the data if the model_type doesn't handle missing data
  if(model_type %in% c(1,3) & !is.null(idealdata@miss_val)) {
    Y <- na_if(Y,idealdata@miss_val)
  }
  
  #Remove NA values, which should have been coded correctly in the make_idealdata function
  
  remove_nas <- !is.na(Y)
  Y <- Y[remove_nas]
  legispoints <- legispoints[remove_nas]
  billpoints <- billpoints[remove_nas]
  timepoints <- timepoints[remove_nas]

  pin_vals <- if(any(is.null(pin_vals))) {
    rep(1,idealdata@restrict_num_high)
  } else {
    pin_vals
  }
  dim(pin_vals) <- idealdata@restrict_num_high
  this_data <- list(N=length(Y),
                    T=max(idealdata@time),
                    Y=Y,
                    hier_type=hier_type,
                    num_legis=num_legis,
                    num_bills=num_bills,
                    ll=legispoints,
                    bb=billpoints,
                    num_fix_high=idealdata@restrict_num_high,
                    num_fix_low=idealdata@restrict_num_low,
                    constrain_par=idealdata@param_fix,
                    constraint_type=idealdata@constraint_type,
                    LX=dim(idealdata@person_cov)[1],
                    SRX=ncol(idealdata@item_cov),
                    SAX=ncol(idealdata@item_cov_miss),
                    legis_pred=idealdata@person_cov,
                    srx_pred=idealdata@item_cov,
                    sax_pred=idealdata@item_cov_miss,
                    time=timepoints,
                    particip=avg_particip,
                    model_type=model_type,
                    pin_vals=pin_vals,
                    discrim_reg_sd=discrim_reg_sd,
                    discrim_abs_sd=discrim_miss_sd,
                    diff_reg_sd=diff_reg_sd,
                    diff_abs_sd=diff_miss_sd,
                    legis_sd=person_sd,
                    restrict_sd=restrict_sd,
                    restrict_low_bar=restrict_low_bar,
                    restrict_high_bar=restrict_high_bar,
                    restrict_alpha=restrict_alpha,
                    restrict_beta=restrict_beta)

  outobj <- sample_model(object=idealdata,nchains=nchains,niters=niters,warmup=warmup,ncores=ncores,
                         this_data=this_data,use_vb=use_vb,...)
  
  outobj@model_type <- model_type
  
  return(outobj)
  
}
