% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_weibmsm.R
\name{sim_weibmsm}
\alias{sim_weibmsm}
\title{Simulate multiple trajectories from an interval-censored multi-state model
with Weibull transition intensities}
\usage{
sim_weibmsm(
  data,
  tmat,
  startprobs,
  exact,
  shape,
  scale,
  censshape,
  censscale,
  n_subj,
  obs_pars,
  true_trajec = FALSE
)
}
\arguments{
\item{data}{A \code{data.frame} or \code{matrix} with named columns \code{time} 
and \code{id}, representing the observation times and corresponding subject id(entifier).}

\item{tmat}{A transition matrix as created by \code{\link[mstate:transMat]{transMat}}, 
with H rows and H columns indicating the states. The total number of possible 
transitions will be indicated by M.}

\item{startprobs}{A numeric vector of length H indicating the probability of 
each subject to start in any of the possible states. Must sum to 1. By default,
all subjects will start in state 1.}

\item{exact}{A numeric vector indicating which states are exactly observed. 
The transition time to exact states will be observed at exact times, regardless 
of the times in \code{obstimes}. No exact states if missing.}

\item{shape}{A numeric vector of length M indicating the shape of the Weibull 
transition intensity for the corresponding transition in \code{tmat}. See 
\code{help(dweibull)}.}

\item{scale}{A numeric vector of length M indicating the scale of the Weibull 
transition intensity for the corresponding transition in \code{tmat}. See 
\code{help(dweibull)}.}

\item{censshape}{A numeric vector of length H indicating the Weibull 
censoring shape in each of the states. If no censoring is required in some states, 
set corresponding entries to \code{NA}. If left missing, 
no censoring is applied. See details.}

\item{censscale}{A numeric vector of length H indicating the Weibull censoring 
scale in each of the states. If no censoring is required in some states, 
set corresponding entries to \code{NA}. If left missing, no censoring is applied.
See details.}

\item{n_subj}{(Optional) Instead of specifying \code{data}, specify the number 
of subjects to generate trajectories for. Requires \code{obs_pars}  to
also be specified.}

\item{obs_pars}{(Optional) A numeric vector of length 3 specifying what the 
time is between planned assessments, what the uniform deviation from this 
time is at the visits and the maximum visit time. 
Specifying \code{obs_pars = c(2, 0.5, 20)} will generate 
a grid of observation times (0, 2, 4, ..., 20) with a uniform[-0.5, 0.5] random 
variable added to each observation time, and cut-off at the end-points 0 and 20.
The observation times may not overlap, 
so the first argument must be at least twice as large as the second.}

\item{true_trajec}{Should the true (right-censored) trajectory be returned for
the subjects as well? Default = \code{FALSE}.}
}
\value{
A \code{matrix} with 3 columns \code{time}, \code{state} and \code{id}, indicating 
the observation time, the corresponding state and subject identifier. If 
\code{true_trajec = TRUE}, a \code{list} with the matrix described above and a matrix 
representing the underlying right-censored trajectory.
}
\description{
Simulate multiple trajectories from a multi-state model quantified
by a transition matrix, with interval-censored transitions and Weibull 
distributed transition intensities. Allows for Weibull censoring in each of 
the states.
}
\details{
Taking \code{(cens)shape} to be 1 for all transitions, we obtain exponential 
(censoring)/transitions with rate 1/\code{(cens)scale}.

If right-censoring parameters are specified, a right-censoring time is generated in 
each of the visited states. If the subject is right-censored, we assume the subject 
is no longer observed at later \code{obstimes}. Due to the interval-censored 
nature of the generation process, it may therefore appear as if the subject 
was right-censored in an earlier state.

Suppose a subject arrives in state g at time s. If we wish to generate 
a survival time from that state according to a Weibull intensity in a clock forward 
model, we can use the inverse transform of the conditional Weibull intensity.
More specifically, letting \eqn{a}{a} denote the shape and \eqn{\sigma}{\sigma} denote the scale, 
the conditional survival function for \eqn{t > s}{t > s} is given by
\deqn{S(t|s) = \mathbf{P}(T \geq t | T \geq s) = \exp(\left( \frac{s}{\sigma} \right)^a - \left( \frac{t}{\sigma} \right)^a)}{S(t|s) = P(T >= t | T >= s) = exp((s/\sigma)^a - (t/\sigma)^a)}
The corresponding cumulative intensity is then given by:
\deqn{A(t|s) = -\log(S(t|s)) = \left( \frac{t}{\sigma} \right)^a - \left( \frac{s}{\sigma} \right)^a}{A(t|s) = - log(S(t|s)) = (t/\sigma)^a - (s/\sigma)^a}
And the inverse cumulative intensity is then:
\deqn{A^{-1}(t|s) = \sigma \sqrt[a]{t + \left( \frac{s}{\sigma} \right)^a}}{A^(-1)(t|s) = \sigma (t + (s/\sigma)^a)^(1/a)}
A conditional survival time is then generated by:
\deqn{T|s = A^{-1}(-\log(U)|s)}{T|s = A^(-1)(-log(U)|s)}
with \eqn{U}{U} a sample from the standard uniform distribution.
If we additionally have covariates (or frailties), the \eqn{-\log(U)}{-log(U)}
above should be replaced by \eqn{\frac{-\log(U)}{\exp(\beta X)}}{(-log(U))/(exp(beta X))}
with \eqn{\beta}{beta} and \eqn{X}{X} the coefficients and covariates respectively.
}
\examples{
require(mstate)
require(ggplot2)
#Generate from an illness-death model with exponential transitions with 
#rates 1/2, 1/10 and 1 for 10 subjects over a time grid.
gd <- sim_weibmsm(tmat = trans.illdeath(), shape = c(1,1,1),
                  scale = c(2, 10, 1), n_subj = 10, obs_pars = c(2, 0.5, 20), 
                  startprobs = c(0.9, 0.1, 0), true_trajec = TRUE)

#Observed trajectories
visualise_msm(gd$observed)
#True trajectories
visualise_msm(gd$true)


#Can supply data-frame with specified observation times
obs_df <- data.frame(time = c(0, 1, 3, 5, 0.5, 6, 9),
                     id = c(1, 1, 1, 1, 2, 2, 2))
gd <- sim_weibmsm(data = obs_df, tmat = trans.illdeath(), shape = c(1, 1, 1),
                  scale = c(2, 10, 1))
visualise_msm(gd)
}
