\name{shakeandbake}
\alias{shakeandbake}
\alias{sab.init}
\alias{sab.run}
\title{
"Shake and Bake" sampler
}
\description{
The "Shake and Bake" method generates a Markov Chain whose stable state converges on the uniform distribution over a the boundary of a convex polytope defined by a set of linear inequality constraints. \code{shakeandbake} further uses the Moore-Penrose pseudo-inverse to eliminate an arbitrary set of linear equality constraints before applying the "Shake and Bake" sampler.

\code{sab.init} and \code{sab.run} together provide a re-entrant version of \code{shakeandbake} so that the Markov chain can be continued if convergence is not satisfactory.
}
\usage{
shakeandbake(constr, n.samples=1E4,
    thin.fn = function(n) { ceiling(log(n + 1)/4 * n^3) }, thin = NULL,
    x0.randomize=FALSE, x0.method="slacklp", x0 = NULL, eliminate = TRUE)

sab.init(constr,
    thin.fn = function(n) { ceiling(log(n + 1)/4 * n^3) }, thin = NULL,
    x0.randomize=FALSE, x0.method="slacklp", x0 = NULL, eliminate = TRUE)

sab.run(state, n.samples)
}
\arguments{
  \item{constr}{Linear constraints that define the sampling space (see details)}
  \item{n.samples}{The desired number of samples to return. The sampler is run for \code{n.samples * thin} iterations}
  \item{thin.fn}{Function that specifies a thinning factor depending on the dimension of the sampling space after equality constraints have been eliminated. Will only be invoked if \code{thin} is \code{NULL}}
  \item{thin}{The thinning factor}
  \item{x0}{Seed point for the Markov Chain. The seed point is specified in the original space, and transformed to the sampling space automatically.}
  \item{x0.method}{Method to generate the seed point if \code{x0} is unspecified, see \code{\link{createSeedPoint}}}
  \item{x0.randomize}{Whether to generate a random seed point if \code{x0} is unspecified}
  \item{eliminate}{Whether to eliminate redundant constraints before constructing the transformation to the sampling space and (optionally) calculating the seed point.}
  \item{state}{A state object, as generated by {\code{har.init}} (see value)}
}
\details{
The constraints are given as a list with the elements \code{constr}, \code{dir} and \code{rhs}. \code{dir} is a vector with values \code{'='} or \code{'<='}. \code{constr} is a matrix and \code{rhs} a vector, which encode the standard linear programming constraint froms \eqn{Ax = b} and \eqn{Ax \leq b} (depending on \code{dir}). The lengths of \code{rhs} and \code{dir} must match the number of rows of \code{constr}.

\code{shakeandbake} applies \code{\link{solution.basis}} to generate a basis of the (translated) solution space of the linear constraints (if any). An affine transformation is generated using \code{\link{createTransform}} and applied to the constraints. Then, a seed point satisfying the inequality constraints is generated using \code{\link{createSeedPoint}}. The closest face to this point is found using \code{\link{findFace}}. Finally, \code{\link{sab}} is used to generate the samples.
}
\value{
For \code{shakeandbake}, a matrix containing the generated samples as rows.

For \code{sab.init}, a state object, containing:
  \item{basis}{The basis for the sampling space. See \code{\link{solution.basis}}.}
  \item{transform}{The sampling space transformation. See \code{\link{createTransform}}.}
  \item{constr}{The linear inequality constraints translated to the sampling space. See \code{\link{transformConstraints}}.}
  \item{x0}{The generated seed point. See \code{\link{createSeedPoint}}.}
  \item{i0}{The index of the closest face. See \code{\link{findFace}}.}
  \item{thin}{The thinning factor to be used.}

For \code{sab.run}, a list containing:
	\item{samples}{A matrix containing the generated samples as rows.}
	\item{state}{A state object that can be used to continue sampling from the Markov chain (i.e. \code{x0} and \code{i0} have been modified).}
}
\author{
Gert van Valkenhoef
}
\note{
"Shake and Bake" is a Markov Chain Monte Carlo (MCMC) method, so generated samples form a correlated time series.
}
\seealso{
\code{\link{harConstraints}}
\code{\link{sab}}
}
\examples{
# Sample from the 3-simplex with the additional constraint that w_1/w_2 = 2
# Three inequality constraints, two equality constraints
constr <- mergeConstraints(simplexConstraints(3), exactRatioConstraint(3, 1, 2, 2))
samples <- shakeandbake(constr, n.samples=1000)
stopifnot(dim(samples) == c(1000, 3))
stopifnot(all.equal(apply(samples, 1, sum), rep(1, 1000)))

sel <- samples[,3] > 0.5 # detect which side we're on
stopifnot(all.equal(samples[sel,], matrix(rep(c(0,0,1), each=sum(sel)), ncol=3)))
stopifnot(all.equal(samples[!sel,], matrix(rep(c(2/3,1/3,0), each=sum(sel)), ncol=3)))

# Sample from the unit rectangle (no equality constraints)
constr <- list(
  constr = rbind(c(1,0), c(0,1), c(-1,0), c(0,-1)),
  dir=rep('<=', 4),
  rhs=c(1, 1, 0, 0))
state <- sab.init(constr)
result <- sab.run(state, n.samples=1000)
faces <- result$faces
samples <- result$samples
stopifnot(all(samples >= -1e-15 & samples <= 1 + 1e-15))

stopifnot(all.equal(samples[faces==1,1], rep(1, sum(faces==1))))
stopifnot(all.equal(samples[faces==2,2], rep(1, sum(faces==2))))
stopifnot(all.equal(samples[faces==3,1], rep(0, sum(faces==3))))
stopifnot(all.equal(samples[faces==4,2], rep(0, sum(faces==4))))

# Continue sampling from the same chain:
result <- sab.run(result$state, n.samples=1000)
samples <- rbind(samples, result$samples)
}
\keyword{shake-and-bake}
\keyword{uniform sampling}
