#' Run one or more Markov Model
#' 
#' Runs one or more unevaluated Markov Models. When more 
#' than one model is provided, all models should have the 
#' same states and state value names.
#' 
#' 
#' A usual situation where more than one model needs to be 
#' run is when comparing different care startegies.
#' 
#' In order to compute comparisons Markov Models must be 
#' similar (same states and state value names). Thus models 
#' should only differ through parameters, transition matrix 
#' cell values and values attached to states (but not state 
#' value names).
#' 
#' The initial number of individuals in each state and the 
#' number of cycle will be the same for all models.
#' 
#' \code{state_cycle_limit} can be specified in 3 different 
#' ways: 1. As a single value: the limit is applied to all 
#' states in all models. 2. As a named vector (where names 
#' are state names): the limits are applied to the given 
#' state names, for all models. 3. As a named list of named 
#' vectors: the limits are applied to the given state names 
#' for the given models.
#' 
#' @param ... One or more \code{uneval_model} object.
#' @param parameters Optional. An object generated by 
#'   \code{\link{define_parameters}}.
#' @param init numeric vector, same length as number of 
#'   model states. Number of individuals in each model state
#'   at the beginning.
#' @param cycles positive integer. Number of Markov Cycles 
#'   to compute.
#' @param cost Names or expression to compute cost on the 
#'   cost-effectiveness plane.
#' @param effect Names or expression to compute effect on 
#'   the cost-effectiveness plane.
#' @param method Counting method.
#' @param uneval_strategy_list List of models, only used by 
#'   \code{run_model_} to avoid using \code{...}.
#' @param state_cycle_limit Optional expansion limit for 
#'   \code{state_cycle}, see details.
#' @param central_strategy The strategy at the center of the
#'   cost-effectiveness plane, for readability.
#'   
#' @return A list of evaluated models with computed values.
#' @export
#' 
#' @example inst/examples/example_run_model.R
#'   
run_model <- function(...,
                      parameters = define_parameters(),
                      init = c(1000L, rep(0L, get_state_number(get_states(list(...)[[1]])) - 1)),
                      cycles = 1,
                      method = c("life-table", "beginning", "end",
                                 "half-cycle"),
                      cost = NULL, effect = NULL,
                      state_cycle_limit = NULL,
                      central_strategy = NULL) {
  
  uneval_strategy_list <- list(...)
  
  method <- match.arg(method)
  
  run_model_(
    uneval_strategy_list = uneval_strategy_list,
    parameters = parameters,
    init = init,
    cycles = cycles,
    method = method,
    cost = lazyeval::lazy_(substitute(cost), env = parent.frame()),
    effect = lazyeval::lazy_(substitute(effect), env = parent.frame()),
    state_cycle_limit = state_cycle_limit,
    central_strategy = central_strategy
  )
}

#' @export
#' @rdname run_model
run_model_ <- function(uneval_strategy_list,
                       parameters,
                       init,
                       cycles,
                       method,
                       cost, effect,
                       state_cycle_limit,
                       central_strategy) {
  
  if (! is.wholenumber(cycles)) {
    stop("'cycles' must be a whole number.")
  }
  
  if (! all(unlist(lapply(
    uneval_strategy_list,
    function(x) "uneval_model" %in% class(x))))) {
    
    .x <- names(uneval_strategy_list[! unlist(lapply(
      uneval_strategy_list,
      function(x) "uneval_model" %in% class(x)))])
    
    stop(sprintf(
      "Incorrect model object%s: %s.",
      plur(length(.x)),
      paste(.x, collapse = ", ")
    ))
  }
  
  list_ce <- list(
    .cost = cost,
    .effect = effect
  )
  
  ce <- c(
    lazyeval::lazy_dots(),
    list_ce
  )
  
  strategy_names <- names(uneval_strategy_list)
  
  if (is.null(strategy_names)) {
    message("No named model -> generating names.")
    strategy_names <- as.character(utils::as.roman(seq_along(uneval_strategy_list)))
    names(uneval_strategy_list) <- strategy_names
  }
  
  if (any(strategy_names == "")) {
    warning("Not all models are named -> generating names.")
    strategy_names <- as.character(utils::as.roman(seq_along(uneval_strategy_list)))
    names(uneval_strategy_list) <- strategy_names
  }
  
  if (! list_all_same(lapply(uneval_strategy_list,
                             function(x) sort(get_state_names(x))))) {
    stop("State names differ between models.")
  }
  
  if (! list_all_same(lapply(uneval_strategy_list,
                             function(x) sort(get_state_value_names(x))))) {
    stop("State value names differ between models.")
  }
  
  if (! length(init) == get_state_number(uneval_strategy_list[[1]])) {
    stop(sprintf(
      "Length of 'init' vector (%i) differs from number of states (%i).",
      length(init),
      get_state_number(uneval_strategy_list[[1]])
    ))
  }
  
  if (! any(init > 0)) {
    stop("At least one init count must be > 0.")
  }
  
  if (is.null(names(init)))
    names(init) <- get_state_names(uneval_strategy_list[[1]])
  
  if (! all(sort(names(init)) == sort(get_state_names(uneval_strategy_list[[1]])))) {
    stop("Names of 'init' vector differ from state names.")
  }
  
  state_cycle_limit <- complete_scl(
    state_cycle_limit,
    state_names = get_state_names(uneval_strategy_list[[1]]),
    strategy_names = names(uneval_strategy_list),
    cycles = cycles
  )
  
  eval_strategy_list <- list()
  
  for (n in names(uneval_strategy_list)) {
    eval_strategy_list[[n]] <- eval_strategy(
      strategy = uneval_strategy_list[[n]], 
      parameters = parameters,
      init = init, 
      cycles = cycles,
      method = method,
      expand_limit = state_cycle_limit[[n]]
    )
  }
  
  list_res <- lapply(eval_strategy_list, get_total_state_values)
  
  for (n in strategy_names){
    list_res[[n]]$.strategy_names <- n
  }
  
  res <- Reduce(dplyr::bind_rows, list_res) %>% 
    dplyr::mutate_(.dots = ce)
  
  root_strategy <- get_root_strategy(res)
  noncomparable_strategy <- get_noncomparable_strategy(res)
  
  if (is.null(central_strategy)) {
    central_strategy <- get_central_strategy(res)
  }
  
  structure(
    list(
      run_model = res,
      eval_strategy_list = eval_strategy_list,
      uneval_strategy_list = uneval_strategy_list,
      parameters = parameters,
      init = init,
      cycles = cycles,
      method = method,
      ce = ce,
      root_strategy = root_strategy,
      central_strategy = central_strategy,
      noncomparable_strategy = noncomparable_strategy
    ),
    class = c("run_model", class(res))
  )
}

get_strategy_names <- function(x) {
  get_model_results(x)$.strategy_names
}

get_strategy_count <- function(x) {
  nrow(get_model_results(x))
}

get_state_value_names.run_model <- function(x) {
  get_state_value_names(x$uneval_strategy_list[[1]])
}

get_total_state_values <- function(x) {
  # faster than as.data.frame or dplyr::as_data_frame
  res <- as.list(colSums((x$values)[- 1]))
  class(res) <- "data.frame"
  attr(res, "row.names") <- c(NA, -1)
  res
}

get_root_strategy <- function(x, ...) {
  UseMethod("get_root_strategy")
}

get_root_strategy.default <- function(x, ...) {
  if (! all(c(".cost", ".effect") %in% names(x))) {
    warning("No cost and/or effect defined, cannot find root strategy.")
    return(invisible(NULL))
  }
  (x %>% 
      dplyr::arrange_(.dots = list(~ .cost, ~ desc(.effect))) %>% 
      dplyr::slice(1))$.strategy_names
}

get_root_strategy.run_model <- function(x, ...) {
  x$root_strategy
}

get_noncomparable_strategy <- function(x, ...) {
  UseMethod("get_noncomparable_strategy")
}

get_noncomparable_strategy.default <- function(x, ...) {
  if (! ".effect" %in% names(x)) {
    warning("No effect defined, cannot find noncomparable strategy.")
    return(invisible(NULL))
  }
  (x %>% 
      dplyr::arrange_(.dots = list(~ .effect)) %>% 
      dplyr::slice(1))$.strategy_names
}

get_noncomparable_strategy.run_model <- function(x, ...) {
  x$noncomparable_strategy
}

get_central_strategy <- function(x, ...) {
  UseMethod("get_central_strategy")
}

get_central_strategy.default <- function(x, ...) {
  get_root_strategy(x)
}

get_central_strategy.run_model <- function(x, ...) {
  x$central_strategy
}

get_effect <- function(x) {
  get_model_results(x)$.effect
}

#' Get Strategy Values
#' 
#' Given a result from \code{\link{run_model}}, return 
#' cost and effect values for a specific strategy.
#' 
#' @param x Result from \code{\link{run_model}}.
#' @param ...	further arguments passed to or from other
#'   methods.
#'   
#' @return A data frame of values per state.
#' @export
get_values <- function(x, ...) {
  UseMethod("get_values")
}

#' @rdname get_values
#' @export
get_values.run_model <- function(x, ...) {
  res <- do.call(
    bind_rows,
    lapply(
      get_strategy_names(x),
      function(.n) {
        get_values(x$eval_strategy_list[[.n]]) %>% 
          dplyr::mutate_(.strategy_names = ~ .n)
      }
    )
  )
  
  tidyr::gather_(
    data = res,
    key_col = "value_names",
    value_col = "value",
    gather_cols = names(res)[! names(res) %in% 
                               c("markov_cycle",
                                 ".strategy_names")]
  )
}

#' @rdname get_values
#' @export
get_values.eval_strategy <- function(x, ...) {
  x$values
}

#' @rdname get_values
#' @export
get_values.list <- function(x, ...) {
  x$values
}

#' Get State Membership Counts
#' 
#' Given a result from \code{\link{run_model}}, return 
#' state membership counts for a specific strategy.
#' 
#' @param x Result from \code{\link{run_model}}.
#' @param ...	further arguments passed to or from other
#'   methods.
#'   
#' @return A data frame of counts per state.
#' @export
get_counts <- function(x, ...) {
  UseMethod("get_counts")
}

#' @rdname get_counts
#' @export
get_counts.run_model <- function(x, ...) {
  res <- do.call(
    bind_rows,
    lapply(
      get_strategy_names(x),
      function(.n) {
        get_counts(x$eval_strategy_list[[.n]]) %>% 
          dplyr::mutate_(
            .strategy_names = ~ .n,
            markov_cycle = ~ row_number())
      }
    )
  )
  
  tidyr::gather_(
    data = res,
    key_col = "state_names",
    value_col = "count",
    gather_cols = names(res)[! names(res) %in% 
                               c("markov_cycle",
                                 ".strategy_names")]
  )
}

#' @rdname get_counts
#' @export
get_counts.eval_strategy <- function(x, ...) {
  x$counts
}

#' @rdname get_counts
#' @export
get_counts.list <- function(x, ...) {
  x$counts
}

get_init <- function(x) {
  UseMethod("get_init")
}

get_init.run_model <- function(x) {
  x$init
}

get_ce <- function(x) {
  x$ce
}

get_parameters <- function(x) {
  x$parameters
}

get_cycles <- function(x) {
  UseMethod("get_cycles")
}

get_cycles.run_model <- function(x) {
  x$cycles
}

get_method <- function(x) {
  UseMethod("get_method")
}
get_method.run_model <- function(x) {
  x$method
}

get_state_names.run_model <- function(x, ...) {
  get_state_names(x$uneval_strategy_list[[1]])
}
