#' Define a Markov Model
#' 
#' Combine information on parameters, transition matrix and 
#' states defined through \code{\link{define_parameters}}, 
#' \code{\link{define_matrix}} and 
#' \code{\link{define_state}} respectively.
#' 
#' This function checks whether the objects are compatible 
#' in the same model (same state names...).
#' 
#' 
#' @param transition_matrix An object generated by 
#'   \code{\link{define_matrix}}.
#' @param ... Object generated by 
#'   \code{\link{define_state}}.
#' @param states List of states, only used by 
#'   \code{define_model_} to avoid using \code{...}.
#'   
#' @return An object of class \code{uneval_model} (a list 
#'   containing the unevaluated parameters, matrix and 
#'   states).
#'   
#' @export
#' 
#' @example inst/examples/example_define_model.R
define_model <- function(...,
                         transition_matrix = define_matrix()) {
  
  states <- define_state_list_(list(...))
  
  define_model_(
    transition_matrix = transition_matrix,
    states = states
  )
}

#' @rdname define_model
#' @export
define_model_ <- function(transition_matrix, states) {
  
  if (! get_state_number(states) == get_matrix_order(transition_matrix)) {
    stop(sprintf(
      "Number of state in model input (%i) differ from number of state in transition matrix (%i).",
      get_state_number(states),
      get_matrix_order(transition_matrix)
    ))
  }
  
  if (! identical(
    sort(get_state_names(states)),
    sort(get_state_names(transition_matrix))
  )) {
    stop("State names in model input differ from transition matrix.")
  }
  
  structure(
    list(
      transition_matrix = transition_matrix,
      states = states
    ), class = "uneval_model")
}

#' Get Markov Model Transition Matrix
#' 
#' Works on both unevaluated and evaluated models.
#' 
#' @param x An \code{uneval_model} or \code{eval_model} 
#'   object.
#'   
#' @return An \code{uneval_matrix} or \code{uneval_matrix} 
#'   object.
#'   
#' @keywords internal
get_matrix <- function(x){
  UseMethod("get_matrix")
}

get_matrix.default <- function(x){
  x$transition_matrix
}

get_states <- function(x){
  UseMethod("get_states")
}

get_states.default <- function(x){
  x$states
}

get_state_value_names.uneval_model <- function(x) {
  get_state_value_names(get_states(x))
}

get_state_names.uneval_model <- function(x, ...) {
  get_state_names(get_states(x))
}
