#' Check Markov Model Transition Matrix
#' 
#' Check whether a matrix fullfills the conditions to be a 
#' transition matrix.
#' 
#' This function is called by \code{\link{eval_matrix}} and 
#' should not be used directly.
#' 
#' Checks whether all rows sum to 1 and all probabilities 
#' are between 0 and 1.
#' 
#' @param x a matrix.
#' @param ... A list of informations to print when checks 
#'   fail, for debugging purposes.
#'   
#' @return \code{NULL}
#'   
#' @keywords internal
check_matrix <- function(x, ...) {
  info <- list(...)
  
  if (! isTRUE(all.equal(rowSums(x), rep(1, nrow(x))))) {
    stop("Not all transition matrix rows sum to 1.")
  }
  
  if (! all(x >= 0 & x <= 1)) {
    stop("Some transition probabilities are outside the interval [0 - 1].")
  }
}

#' Evaluate Markov Model Transition Matrix
#' 
#' Evaluate a transition matrix using evaluated parameters.
#' 
#' Runs checks on the transition matrix during evaluation.
#' 
#' @param x an \code{uneval_matrix} object.
#' @param parameters an \code{eval_parameters} object.
#'   
#' @return An \code{eval_matrix} object (actually a list of 
#'   transition matrix, one per cycle).
#'   
#' @keywords internal
eval_matrix <- function(x, parameters) {

  tab_res <- mutate_(parameters, C = -pi, .dots = x)[names(x)]
  
  n <- get_matrix_order(x)

  f3 <- function(this_array) {
    posC <- this_array == -pi
    
    stopifnot(
      all(rowSums(posC, dims = 2) <= 1)
    )
    this_array[posC] <- 0
    
    valC <- 1 - rowSums(this_array, dims = 2)[which(posC, arr.ind = TRUE)[, -3]] 
    this_array[posC] <- valC
    
    sapply(1:dim(this_array)[1], function(i){check_matrix(this_array[i,,])})
    this_array
  }
  
  tab_res_2 <- array(unlist(tab_res), dim = c(nrow(tab_res), n, n))
  
  for(i in 1:nrow(tab_res)){
    tab_res_2[i,,] <- t(tab_res_2[i,,])
  }
  
  tab_res_3 <- f3(tab_res_2)
  tab_res_4 <- split_along_dim(tab_res_3, 1)
  
  structure(tab_res_4,
            class = c("eval_matrix", class(tab_res_4)),
            state_names = get_state_names(x))
}


split_along_dim <- function(a, n)
  setNames(lapply(split(a, arrayInd(seq_along(a), dim(a))[, n]),
                  array, dim = dim(a)[-n], dimnames(a)[-n]),
           dimnames(a)[[n]])


get_state_names.eval_matrix <- function(x, ...){
  attr(x, "state_names")
}

get_matrix_order.eval_matrix <- function(x){
  ncol(x[[1]])
}
