/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tpkg.h"     

static herr_t H5T__set_precision(const H5T_t *dt, size_t prec);

size_t
H5Tget_precision(hid_t type_id)
{
    H5T_t *dt;
    size_t ret_value;

    FUNC_ENTER_API(0)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, 0, "not a datatype");

    
    if ((ret_value = H5T_get_precision(dt)) == 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, 0, "can't get precision for specified datatype");

done:
    FUNC_LEAVE_API(ret_value)
} 

size_t
H5T_get_precision(const H5T_t *dt)
{
    size_t ret_value = 0; 

    FUNC_ENTER_NOAPI(0)

    
    while (dt->shared->parent)
        dt = dt->shared->parent;
    if (!H5T_IS_ATOMIC(dt->shared))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, 0, "operation not defined for specified datatype");

    
    ret_value = dt->shared->u.atomic.prec;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Tset_precision(hid_t type_id, size_t prec)
{
    H5T_t *dt        = NULL;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is read-only");
    if (NULL != dt->vol_obj)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTSET, FAIL, "datatype is committed");
    if (prec == 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "precision must be positive");
    if (H5T_ENUM == dt->shared->type && dt->shared->u.enumer.nmembs > 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "operation not allowed after members are defined");
    if (H5T_STRING == dt->shared->type)
        HGOTO_ERROR(H5E_ARGS, H5E_UNSUPPORTED, FAIL, "precision for this type is read-only");
    if (H5T_COMPOUND == dt->shared->type || H5T_OPAQUE == dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "operation not defined for specified datatype");

    
    if (H5T__set_precision(dt, prec) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "unable to set precision");

done:
    FUNC_LEAVE_API(ret_value)
}

static herr_t
H5T__set_precision(const H5T_t *dt, size_t prec)
{
    size_t offset, size;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dt);
    assert(prec > 0);
    assert(H5T_OPAQUE != dt->shared->type);
    assert(H5T_COMPOUND != dt->shared->type);
    assert(H5T_STRING != dt->shared->type);
    assert(!(H5T_ENUM == dt->shared->type && 0 == dt->shared->u.enumer.nmembs));

    if (dt->shared->parent) {
        if (H5T__set_precision(dt->shared->parent, prec) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSET, FAIL, "unable to set precision for base type");

        
        if (dt->shared->type == H5T_ARRAY)
            dt->shared->size = dt->shared->parent->shared->size * dt->shared->u.array.nelem;
        else if (dt->shared->type == H5T_COMPLEX)
            dt->shared->size = 2 * dt->shared->parent->shared->size;
        else if (dt->shared->type != H5T_VLEN)
            dt->shared->size = dt->shared->parent->shared->size;
    }
    else {
        if (H5T_IS_ATOMIC(dt->shared)) {
            
            offset = dt->shared->u.atomic.offset;
            size   = dt->shared->size;
            if (prec > 8 * size)
                offset = 0;
            else if (offset + prec > 8 * size)
                offset = 8 * size - prec;
            if (prec > 8 * size)
                size = (prec + 7) / 8;

            
            switch (dt->shared->type) {
                case H5T_INTEGER:
                case H5T_TIME:
                case H5T_BITFIELD:
                    
                    break;

                case H5T_FLOAT:
                    
                    if (dt->shared->u.atomic.u.f.sign >= prec + offset ||
                        dt->shared->u.atomic.u.f.epos + dt->shared->u.atomic.u.f.esize > prec + offset ||
                        dt->shared->u.atomic.u.f.mpos + dt->shared->u.atomic.u.f.msize > prec + offset)
                        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL,
                                    "adjust sign, mantissa, and exponent fields first");
                    break;

                case H5T_NO_CLASS:
                case H5T_STRING:
                case H5T_OPAQUE:
                case H5T_COMPOUND:
                case H5T_REFERENCE:
                case H5T_ENUM:
                case H5T_VLEN:
                case H5T_ARRAY:
                case H5T_COMPLEX:
                case H5T_NCLASSES:
                default:
                    HGOTO_ERROR(H5E_ARGS, H5E_UNSUPPORTED, FAIL, "operation not defined for datatype class");
            } 

            
            dt->shared->size            = size;
            dt->shared->u.atomic.offset = offset;
            dt->shared->u.atomic.prec   = prec;
        } 
        else
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "operation not defined for specified datatype");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
}
