/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the COPYING file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#ifndef H5Mpublic_H
#define H5Mpublic_H
#include "H5public.h"
#include "H5Ipublic.h"
#include "H5VLpublic.h"
#include "H5VLconnector.h"
#define H5VL_MAP_CREATE 1
#define H5VL_MAP_OPEN 2
#define H5VL_MAP_GET_VAL 3
#define H5VL_MAP_EXISTS 4
#define H5VL_MAP_PUT 5
#define H5VL_MAP_GET 6
#define H5VL_MAP_SPECIFIC 7
#define H5VL_MAP_OPTIONAL 8
#define H5VL_MAP_CLOSE 9
typedef enum H5VL_map_get_t {
    H5VL_MAP_GET_MAPL,
    H5VL_MAP_GET_MCPL,
    H5VL_MAP_GET_KEY_TYPE,
    H5VL_MAP_GET_VAL_TYPE,
    H5VL_MAP_GET_COUNT
} H5VL_map_get_t;
typedef enum H5VL_map_specific_t {
    H5VL_MAP_ITER,
    H5VL_MAP_DELETE
} H5VL_map_specific_t;
//! <!-- [H5M_iterate_t_snip] -->
typedef herr_t (*H5M_iterate_t)(hid_t map_id, const void *key, void *op_data);
//! <!-- [H5M_iterate_t_snip] -->
typedef union H5VL_map_args_t {
    struct {
        H5VL_loc_params_t loc_params;
        const char       *name;
        hid_t             lcpl_id;
        hid_t             key_type_id;
        hid_t             val_type_id;
        hid_t             mcpl_id;
        hid_t             mapl_id;
        void             *map;
    } create;
    struct {
        H5VL_loc_params_t loc_params;
        const char       *name;
        hid_t             mapl_id;
        void             *map;
    } open;
    struct {
        hid_t       key_mem_type_id;
        const void *key;
        hid_t       value_mem_type_id;
        void       *value;
    } get_val;
    struct {
        hid_t       key_mem_type_id;
        const void *key;
        hbool_t     exists;
    } exists;
    struct {
        hid_t       key_mem_type_id;
        const void *key;
        hid_t       value_mem_type_id;
        const void *value;
    } put;
    struct {
        H5VL_map_get_t get_type;
        union {
            struct {
                hid_t mapl_id;
            } get_mapl;
            struct {
                hid_t mcpl_id;
            } get_mcpl;
            struct {
                hid_t type_id;
            } get_key_type;
            struct {
                hid_t type_id;
            } get_val_type;
            struct {
                hsize_t count;
            } get_count;
        } args;
    } get;
    struct {
        H5VL_map_specific_t specific_type;
        union {
            struct {
                H5VL_loc_params_t loc_params;
                hsize_t           idx;
                hid_t             key_mem_type_id;
                H5M_iterate_t     op;
                void             *op_data;
            } iterate;
            struct {
                H5VL_loc_params_t loc_params;
                hid_t             key_mem_type_id;
                const void       *key;
            } del;
        } args;
    } specific;
} H5VL_map_args_t;
#ifdef __cplusplus
extern "C" {
#endif
#ifdef H5_HAVE_MAP_API
H5_DLL hid_t H5Mcreate(hid_t loc_id, const char *name, hid_t key_type_id, hid_t val_type_id, hid_t lcpl_id,
                       hid_t mcpl_id, hid_t mapl_id);
#ifndef H5_DOXYGEN
H5_DLL hid_t H5Mcreate_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                             const char *name, hid_t key_type_id, hid_t val_type_id, hid_t lcpl_id,
                             hid_t mcpl_id, hid_t mapl_id, hid_t es_id);
#else
H5_DLL hid_t H5Mcreate_async(hid_t loc_id, const char *name, hid_t key_type_id, hid_t val_type_id,
                             hid_t lcpl_id, hid_t mcpl_id, hid_t mapl_id, hid_t es_id);
#endif
H5_DLL hid_t H5Mcreate_anon(hid_t loc_id, hid_t key_type_id, hid_t val_type_id, hid_t mcpl_id, hid_t mapl_id);
H5_DLL hid_t H5Mopen(hid_t loc_id, const char *name, hid_t mapl_id);
#ifndef H5_DOXYGEN
H5_DLL hid_t H5Mopen_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                           const char *name, hid_t mapl_id, hid_t es_id);
#else
H5_DLL hid_t H5Mopen_async(hid_t loc_id, const char *name, hid_t mapl_id, hid_t es_id);
#endif
H5_DLL herr_t H5Mclose(hid_t map_id);
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Mclose_async(const char *app_file, const char *app_func, unsigned app_line, hid_t map_id,
                             hid_t es_id);
#else
H5_DLL herr_t H5Mclose_async(hid_t map_id, hid_t es_id);
#endif
H5_DLL hid_t H5Mget_key_type(hid_t map_id);
H5_DLL hid_t H5Mget_val_type(hid_t map_id);
H5_DLL hid_t H5Mget_create_plist(hid_t map_id);
H5_DLL hid_t H5Mget_access_plist(hid_t map_id);
H5_DLL herr_t H5Mget_count(hid_t map_id, hsize_t *count, hid_t dxpl_id);
H5_DLL herr_t H5Mput(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id,
                     const void *value, hid_t dxpl_id);
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Mput_async(const char *app_file, const char *app_func, unsigned app_line, hid_t map_id,
                           hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id, const void *value,
                           hid_t dxpl_id, hid_t es_id);
#else
H5_DLL herr_t H5Mput_async(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id,
                           const void *value, hid_t dxpl_id, hid_t es_id);
#endif
H5_DLL herr_t H5Mget(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id, void *value,
                     hid_t dxpl_id);
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Mget_async(const char *app_file, const char *app_func, unsigned app_line, hid_t map_id,
                           hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id, void *value,
                           hid_t dxpl_id, hid_t es_id);
#else
H5_DLL herr_t H5Mget_async(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t val_mem_type_id,
                           void *value, hid_t dxpl_id, hid_t es_id);
#endif
H5_DLL herr_t H5Mexists(hid_t map_id, hid_t key_mem_type_id, const void *key, hbool_t *exists, hid_t dxpl_id);
H5_DLL herr_t H5Miterate(hid_t map_id, hsize_t *idx, hid_t key_mem_type_id, H5M_iterate_t op, void *op_data,
                         hid_t dxpl_id);
H5_DLL herr_t H5Miterate_by_name(hid_t loc_id, const char *map_name, hsize_t *idx, hid_t key_mem_type_id,
                                 H5M_iterate_t op, void *op_data, hid_t dxpl_id, hid_t lapl_id);
H5_DLL herr_t H5Mdelete(hid_t map_id, hid_t key_mem_type_id, const void *key, hid_t dxpl_id);
/// \cond DEV
#ifndef H5M_MODULE
#define H5Mcreate_async(...) H5Mcreate_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Mopen_async(...)   H5Mopen_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Mclose_async(...)  H5Mclose_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Mput_async(...)    H5Mput_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Mget_async(...)    H5Mget_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Mcreate_async_wrap H5_NO_EXPAND(H5Mcreate_async)
#define H5Mopen_async_wrap   H5_NO_EXPAND(H5Mopen_async)
#define H5Mclose_async_wrap  H5_NO_EXPAND(H5Mclose_async)
#define H5Mput_async_wrap    H5_NO_EXPAND(H5Mput_async)
#define H5Mget_async_wrap    H5_NO_EXPAND(H5Mget_async)
#endif
/// \endcond
#ifndef H5_NO_DEPRECATED_SYMBOLS
#endif
#endif
#ifdef __cplusplus
}
#endif
#endif
