% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EK_Tobit.R
\name{EK_Tobit}
\alias{EK_Tobit}
\title{Eaton and Kortum (2001) Tobit model, EK_Tobit}
\usage{
EK_Tobit(y, dist, x, vce_robust = TRUE, data, ...)
}
\arguments{
\item{y}{name (type: character) of the dependent variable in the dataset 
\code{data}, e.g. trade flows.
The variable is logged and then taken as the dependent variable in 
the regression. As the log of zero is not defined, 
all flows equal to zero are replaced by
a left open interval with the logged minimum trade flow of the
respective importing country as right border.}

\item{dist}{name (type: character) of the distance variable in the dataset 
\code{data} containing a measure of distance between all pairs of bilateral
partners. It is logged automatically when the function is executed.}

\item{x}{vector of names (type: character) of those bilateral variables in 
the dataset \code{data} that should be taken as the independent variables 
in the estimation. If an independent variable is a dummy variable,
it should be of type numeric (0/1) in the dataset. If an independent variable 
is defined as a ratio, it should be logged. 
Unilateral variables such as country dummies or incomes can be added. 
If unilateral metric variables such as GDPs should be used as independent 
variables, those variables have to be logged first and the 
logged variable can be used in \code{x}.
Interaction terms can be added.}

\item{vce_robust}{robust (type: logic) determines whether a robust 
variance-covariance matrix should be used. The default is set to \code{TRUE}. 
If set \code{TRUE} the estimation results are consistent with the 
Stata code provided at the website
\href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook}
when choosing robust estimation.}

\item{data}{name of the dataset to be used (type: character). 
To estimate gravity equations, a square gravity dataset including bilateral 
flows defined by the argument \code{y}, ISO-codes of type character 
(called \code{iso_o} for the country of origin and \code{iso_d} for the 
destination country), a distance measure defined by the argument \code{dist} 
and other potential influences given as a vector in \code{x} are required. 
All dummy variables should be of type numeric (0/1). Missing trade flows as 
well as incomplete rows should be excluded from the dataset. 
Zero trade flows are allowed.}

\item{...}{additional arguments to be passed to \code{EK_Tobit}.}
}
\value{
The function returns the summary of the estimated gravity model as a 
\code{\link[survival]{survreg}}-object.
}
\description{
\code{EK_Tobit} estimates gravity models in their additive form
by conducting a censored regression.
It follows the Eaton and Kortum (2001) Tobit model where each country 
is assigned specific ceonsoring bounds.
}
\details{
\code{EK_Tobit} represents the Eaton and Kortum (2001) Tobit model.
When taking the log of the gravity equation flows equal to zero
constitute a problem as their log is not defined.
Therefore, in \code{EK_Tobit} all values of the dependent variable 
are redefined as intervals.
The positive observations have both interval bounds equal
to their original value. 
For zero flows the interval is left open. The right border
of the interval is set to the log of the minimum positive trade flow of 
the respective importing country.
The defined data object of class \code{\link[survival]{Surv}} is
then inserted in \code{\link[survival]{survreg}} for the 
parameter estimation.
 
To execute the function a square gravity dataset with all pairs of 
countries, ISO-codes for the country of origin and destination, a measure of 
distance between the bilateral partners as well as all 
information that should be considered as dependent an independent 
variables is needed. 
Missing bilateral flows as well as incomplete rows should be 
excluded from the dataset.  
Zero trade flows are allowed. 

\code{EK_Tobit} is designed to be consistent with the Stata code provided at
the website
\href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook}
when choosing robust estimation.

Up to now, the function is designed for cross-sectional data,
but can be extended to panel data using the 
\code{\link[survival]{survreg}} function.

For other Tobit functions, see \code{\link[gravity]{Tobit}}
for a simple Tobit model where number \code{1} is added to all observations
and \code{\link[gravity]{ET_Tobit}} for the Eaton and Tamura (1994) 
threshold Tobit model where instead of simply adding number \code{1} 
to the data the threshold is estimated.
}
\examples{
\dontrun{
# Example for data with zero trade flows
data(Gravity_zeros)

Gravity_zeros$lgdp_o <- log(Gravity_zeros$gdp_o)
Gravity_zeros$lgdp_d <- log(Gravity_zeros$gdp_d)

EK_Tobit(y="flow", dist="distw", x=c("rta","lgdp_o","lgdp_d"), 
vce_robust=TRUE, data=Gravity_zeros)

EK_Tobit(y="flow", dist="distw", x=c("rta","iso_o","iso_d"), 
vce_robust=TRUE, data=Gravity_zeros)
}

\dontshow{
# examples for CRAN checks:
# executable in < 5 sec together with the examples above
# not shown to users

data(Gravity_zeros)
Gravity_zeros$lgdp_o <- log(Gravity_zeros$gdp_o)
Gravity_zeros$lgdp_d <- log(Gravity_zeros$gdp_d)

# choose exemplarily 10 biggest countries for check data
countries_chosen_zeros <- names(sort(table(Gravity_zeros$iso_o), decreasing = TRUE)[1:10])
grav_small_zeros <- Gravity_zeros[Gravity_zeros$iso_o \%in\% countries_chosen_zeros,]
EK_Tobit(y="flow", dist="distw", x=c("rta","lgdp_o","lgdp_d"), vce_robust=TRUE, data=grav_small_zeros)
}

}
\references{
For more information on gravity models, theoretical foundations and
estimation methods in general see 

Anderson, J. E. (1979) <DOI:10.12691/wjssh-2-2-5>

Anderson, J. E. (2010) <DOI:10.3386/w16576>

Anderson, J. E. and van Wincoop, E. (2003) <DOI:10.3386/w8079> 

Baier, S. L. and Bergstrand, J. H. (2009) <DOI:10.1016/j.jinteco.2008.10.004>

Baier, S. L. and Bergstrand, J. H. (2010) in Van Bergeijk, P. A., & Brakman, S. (Eds.) (2010) chapter 4 <DOI:10.1111/j.1467-9396.2011.01000.x>

Head, K., Mayer, T., & Ries, J. (2010) <DOI:10.1016/j.jinteco.2010.01.002>

Head, K. and Mayer, T. (2014) <DOI:10.1016/B978-0-444-54314-1.00003-3>

Santos-Silva, J. M. C. and Tenreyro, S. (2006) <DOI:10.1162/rest.88.4.641> 

and the citations therein.


Especially for Tobit models see

Tobin, J. (1958) <DOI:10.2307/1907382>

Eaton, J., & Tamura, A. (1994) <DOI:10.3386/w4758>

Eaton, J., & Kortum, S. (2001) <DOI:10.3386/w8070>.


See Carson, R. T., & Sun, Yixiao (2007) <DOI:10.1111/j.1368-423X.2007.00218.x>
for the estimation of the threshold in a Tobit model. 

See \href{https://sites.google.com/site/hiegravity/}{Gravity Equations: Workhorse, Toolkit, and Cookbook} for gravity datasets and Stata code for estimating gravity models.
}
\seealso{
\code{\link[survival]{Surv}}, \code{\link[survival]{survreg}}
}
