% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_target.R
\name{simulate_target}
\alias{simulate_target}
\alias{generate_target}
\title{Generating time series with controllable features using MAR models}
\usage{
simulate_target(
  length = 100,
  seasonal_periods = 1,
  feature_function,
  target,
  k = ifelse(length(seasonal_periods) == 1, 3, length(seasonal_periods)),
  tolerance = 0.05,
  trace = FALSE,
  parallel = FALSE
)

generate_target(
  length = 100,
  nseries = 10,
  seasonal_periods = 1,
  feature_function,
  target,
  k = ifelse(length(seasonal_periods) == 1, 3, length(seasonal_periods)),
  tolerance = 0.05,
  trace = FALSE,
  parallel = FALSE
)
}
\arguments{
\item{length}{length of the time series to be generated.}

\item{seasonal_periods}{Either a scalar or a numeric vector of length k containing
the number of seasonal periods for each component.}

\item{feature_function}{a function that returns a vector of features from a time series.}

\item{target}{target feature values of the same length as that returned by \code{feature_function()}.}

\item{k}{integer specifying number of components to use for MAR models. Default is 3
unless there are multiple seasonal periods specified.}

\item{tolerance}{average tolerance per feature. The genetic algorithm will attempt
to find a solution where the average difference between the series features and target
is less than \code{tolerance}. A larger value will give a faster but less precise
solution.}

\item{trace}{logical indicating if details of the search should be shown.}

\item{parallel}{An optional argument which allows to specify if the Genetic Algorithm
should be run sequentially or in parallel.}

\item{nseries}{Number of series to generate}
}
\value{
A time series object of class "ts" or "msts".
}
\description{
\code{simulate_target} simulate one time series of length `length` from a MAR model with target features 
and returns a \code{ts} or \code{msts} object.
\code{generate_target} simulate multiple time series of length `length` from a MAR model with target features 
and returns a \code{tsibble} object. The index of the tsibble is guessed from the seasonal periods.
The specified features should not depend on the scale of the time series as the series is scaled during simulation.
}
\examples{
set.seed(1)
library(tsfeatures)
my_features <- function(y) {
  c(entropy(y), acf = acf(y, plot = FALSE)$acf[2:3, 1, 1])
}
# Simulate a ts with specified target features
y <- simulate_target(
  length = 60, feature_function = my_features, target = c(0.5, 0.9, 0.8)
)
my_features(y)
plot(y)
\dontrun{
# Generate a tsibble with specified target features
df <- generate_target(
  length = 60, feature_function = my_features, target = c(0.5, 0.9, 0.8)
)
df \%>\% 
 as_tibble() \%>\%
 group_by(key) \%>\%
 dplyr::summarise(value = my_features(value), 
           feature=c("entropy","acf1", "acf2"),
           .groups = "drop")
autoplot(df)
# Simulate time series similar to an existing series
my_features <- function(y) {
  c(stl_features(y)[c("trend", "seasonal_strength", "peak", "trough")])
}
y <- simulate_target(
  length = length(USAccDeaths),
  seasonal_periods = frequency(USAccDeaths),
  feature_function = my_features, target = my_features(USAccDeaths)
)
tsp(y) <- tsp(USAccDeaths)
plot(cbind(USAccDeaths, y))
cbind(my_features(USAccDeaths), my_features(y))}
}
\author{
Yanfei Kang and Rob J Hyndman
}
