% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prediction.R
\name{prediction}
\alias{prediction}
\title{Perform predictive inference in a Gaussian mixture dynamic Bayesian network}
\usage{
prediction(
  gmdbn,
  evid,
  evid_pred = NULL,
  nodes = names(gmdbn$b_1),
  col_seq = NULL,
  horizon = 1,
  n_part = 1000,
  max_part_sim = 1e+06,
  min_ess = 1,
  verbose = FALSE
)
}
\arguments{
\item{gmdbn}{An object of class \code{gmdbn}.}

\item{evid}{A data frame containing the evidence. Its columns must explicitly
be named after nodes of \code{gmdbn} and can contain missing values (columns
with no value can be removed).}

\item{evid_pred}{A data frame containing the "predicted" evidence. Its
columns must explicitly be named after nodes of \code{gmdbn} and can contain
missing values (columns with no value can be removed).}

\item{nodes}{A character vector containing the inferred nodes (by default all
the nodes of \code{gmdbn}).}

\item{col_seq}{A character vector containing the column names of \code{evid}
and \code{evid_pred} that describe the observation sequence. If \code{NULL}
(the default), all the observations belong to a single sequence. The
observations of a same sequence must be ordered such that the \eqn{t}th one
is related to time slice \eqn{t} (note that the sequences can have different
lengths).}

\item{horizon}{A positive integer vector containing the time horizons for
which predictive inference is performed.}

\item{n_part}{A positive integer corresponding to the number of particles
generated for each observation sequence.}

\item{max_part_sim}{An integer greater than or equal to \code{n_part}
corresponding to the maximum number of particles that can be processed
simultaneously. This argument is used to prevent memory overflow, dividing
\code{evid} into smaller subsets that are handled sequentially.}

\item{min_ess}{A numeric value in [0, 1] corresponding to the minimum ESS
(expressed as a proportion of \code{n_part}) under which the renewal step of
sequential importance resampling is performed. If \code{1} (the default),
this step is performed at each time slice.}

\item{verbose}{A logical value indicating whether subsets of \code{evid} and
time slices in progress are displayed.}
}
\value{
If \code{horizon} has one element, a data frame with a structure
similar to \code{evid} containing the predicted values of the inferred
nodes and their observation sequences (if \code{col_seq} is not \code{NULL}).
If \code{horizon} has two or more elements, a list of data frames (tibbles)
containing these values for each time horizon.
}
\description{
This function performs predictive inference in a Gaussian mixture dynamic
Bayesian network. For a sequence of \eqn{T} time slices, this task consists
in defining a time horizon \eqn{h} such that at each time slice \eqn{t}
(for \ifelse{html}{\out{0 &le; <i>t</i> &le; <i>t</i> &minus;
<i>h</i>}}{\eqn{0 \le t \le T - h}}), the state of the system at \eqn{t + h}
is estimated given all the data (the evidence) collected up to \eqn{t}.
Although the states at \ifelse{html}{\out{<i>t</i> + 1, &hellip; ,
<i>t</i> + <i>h</i>}}{\eqn{t + 1, \dots , t + h}} are observed in the future,
some information about them can be known a priori (such as contextual
information or features controlled by the user). This "predicted" evidence
can be taken into account when propagating the particles from \eqn{t} to
\eqn{t + h} in order to improve the predictions. Predictive inference is
performed by sequential importance resampling, which is a particle-based
approximate method (Koller and Friedman, 2009).
}
\examples{
\donttest{
set.seed(0)
data(gmdbn_air, data_air)
evid <- data_air
evid$NO2[sample.int(7680, 1536)] <- NA
evid$O3[sample.int(7680, 1536)] <- NA
pred <- prediction(gmdbn_air, evid, evid[, c("DATE", "TEMP", "WIND")],
                   nodes = c("NO2", "O3"), col_seq = "DATE",
                   horizon = c(1, 2), verbose = TRUE)}

}
\references{
Koller, D. and Friedman, N. (2009). \emph{Probabilistic Graphical Models:
Principles and Techniques}. The MIT Press.
}
\seealso{
\code{\link{filtering}}, \code{\link{inference}},
\code{\link{smoothing}}
}
