% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scale-continuous-logFC.r
\name{scale_x_logFC}
\alias{scale_x_logFC}
\alias{scale_y_logFC}
\title{Position scales for log fold change data}
\usage{
scale_x_logFC(
  name = "Abundance of x\%unit",
  breaks = NULL,
  labels = NULL,
  limits = symmetric_limits,
  oob = scales::squish,
  expand = expansion(mult = 0.15, add = 0),
  log.base.labels = FALSE,
  log.base.data = 2L,
  ...
)

scale_y_logFC(
  name = "Abundance of y\%unit",
  breaks = NULL,
  labels = NULL,
  limits = symmetric_limits,
  oob = scales::squish,
  expand = expansion(mult = 0.15, add = 0),
  log.base.labels = FALSE,
  log.base.data = 2L,
  ...
)
}
\arguments{
\item{name}{The name of the scale without units, used for the axis-label.}

\item{breaks}{The positions of ticks or a function to generate them. Default
varies depending on argument passed to \code{log.base.labels}. if supplied
as a numeric vector they should be given using the data as passed to
parameter \code{data}.}

\item{labels}{The tick labels or a function to generate them from the tick
positions. The default is function that uses the arguments passed to
\code{log.base.data} and \code{log.base.labels} to generate suitable
labels.}

\item{limits}{limits    One of: NULL to use the default scale range from
ggplot2. A numeric
vector of length two providing limits of the scale, using NA to refer to the
existing minimum or maximum. A function that accepts the existing
(automatic) limits and returns new limits. The default is function
\code{symmetric_limits()} which keep 1 at the middle of the axis..}

\item{oob}{Function that handles limits outside of the scale limits (out of
bounds). The default squishes out-of-bounds values to the boundary.}

\item{expand}{Vector of range expansion constants used to add some padding
around the data, to ensure that they are placed some distance away from
the axes. The default is to expand the scale by 15\% on each end for
log-fold-data, so as to leave space for counts annotations.}

\item{log.base.labels, log.base.data}{integer or logical Base of logarithms used to
express fold-change values in tick labels and in \code{data}. Use \code{FALSE}
for no logarithm transformation.}

\item{...}{other named arguments passed to \code{scale_y_continuous}.}
}
\description{
Continuous scales for x and y aesthetics with defaults suitable for values
expressed as log2 fold change in \code{data} and fold-change in tick labels.
Supports tick labels and data expressed in any combination of fold-change,
log2 fold-change and log10 fold-change. Supports addition of units to
axis labels passed as argument to the \code{name} formal parameter.
}
\details{
These scales only alter default arguments of
  \code{scale_x_continuous()} and \code{scale_y_continuous()}. Please, see
  documentation for \code{\link[ggplot2]{scale_continuous}} for details. The
  name argument supports the use of \code{"\%unit"} at the end of the string to
  automatically add a units string, otherwise user-supplied values for
  names, breaks, and labels work as usual. Tick labels are built based on
  the transformation already applied to the data (log2 by default) and
  apossibly different log transformation (default is fold-change with no
  transformation).
}
\examples{

set.seed(12346)
my.df <- data.frame(x = rnorm(50, sd = 4), y = rnorm(50, sd = 4))

ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC() +
  scale_y_logFC()

ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC(labels = scales::trans_format(function(x) {log10(2^x)},
                         scales::math_format())) +
  scale_y_logFC(labels = scales::trans_format(function(x) {log10(2^x)},
                         scales::math_format()))

ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC(log.base.labels = 2) +
  scale_y_logFC(log.base.labels = 2)

ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC("A concentration\%unit", log.base.labels = 10) +
  scale_y_logFC("B concentration\%unit", log.base.labels = 10)

ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC("A concentration\%unit", breaks = NULL) +
  scale_y_logFC("B concentration\%unit", breaks = NULL)

# taking into account that data are expressed as log2 FC.
ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC("A concentration\%unit", breaks = log2(c(1/100, 1, 100))) +
  scale_y_logFC("B concentration\%unit", breaks = log2(c(1/100, 1, 100)))

ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC(labels = scales::trans_format(function(x) {log10(2^x)},
                         scales::math_format())) +
  scale_y_logFC(labels = scales::trans_format(function(x) {log10(2^x)},
                         scales::math_format()))

# override "special" default arguments.
ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC("A concentration",
                breaks = waiver(),
                labels = waiver()) +
  scale_y_logFC("B concentration",
                breaks = waiver(),
                labels = waiver())

ggplot(my.df, aes(x, y)) +
  geom_point() +
  scale_x_logFC() +
  scale_y_logFC() +
  geom_quadrant_lines() +
  stat_quadrant_counts(size = 3.5)

}
\seealso{
Other scales for omics data: 
\code{\link{outcome2factor}()},
\code{\link{scale_shape_outcome}()},
\code{\link{xy_outcomes2factor}()}
}
\concept{scales for omics data}
