% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utility_functions.R
\name{fsr_components}
\alias{fsr_components}
\alias{create_component}
\alias{component_from_sfg}
\title{Creation of a component}
\usage{
create_component(raw_obj, md, type)

component_from_sfg(sfg, md)
}
\arguments{
\item{raw_obj}{A vector, list or matrix containing the pairs of coordinates  to create the \code{sfg} object of the component.}

\item{md}{A numeric value indicating the membership degree of the component. It has to be a value in \eqn{]0, 1]}.}

\item{type}{A character value that indicates the type of the desired \code{sfg} object.
It should be either \code{"POINT"}, \code{"LINE"}, or \code{"REGION"}.}

\item{sfg}{An \code{sfg} object. It should be either \code{POINT}, \code{MULTIPOINT}, \code{LINESTRING},
\code{MULTILINESTRING}, \code{POLYGON} or \code{MULTIPOLYGON} type. Other types of spatial objects are not allowed.}
}
\value{
A \code{component} object that can be added to a spatial plateau object (i.e., a \code{pgeometry} object).
}
\description{
There are two functions that build a component from coordinate pairs or a
single \code{sfg} object labeled with a membership degree. This component can be added to a spatial plateau object.
A component consists of an \code{sfg} object and an associated membership degree.
A component can be built in two different ways. By using the function \code{create_component}, the component is formed by
the means of a  numeric vector, list or matrix that represents a pair of coordinates.
By using the function \code{component_from_sfg}, the component is created from an \code{sfg}
object.
}
\details{
These functions create a \code{component} object, which is a pair of an \code{sfg} object and a membership degree in \eqn{]0, 1]}.

The function \code{create_component} receives  three parameters: \code{raw_obj}, \code{md} and \code{type}. The use of \code{raw_obj} is
similar to the parameter of the family of functions of the \code{sf} package (\code{st} family) that creates spatial objects
from a numeric vector, matrix or list (e.g., the functions \code{st_point}, \code{st_multipoint}, etc.). The spatial data type
(i.e., the type of the \code{sfg} object) indicated by the parameter \code{type} represents simple and complex objects.
For instance, \code{"POINT"} may refer to simple or complex point objects (internally, we can create a POINT or MULTIPOINT object).

The \code{component_from_sfg} builds a \code{component} object by using the specification of two parameters that directly represents the
pair of an \code{sfg} object and its corresponding membership degree (i.e.,  \code{md} value).
}
\examples{

# Creating two components of the type POINT
v1 = rbind(c(1,2), c(3,4))
v2 = rbind(c(1,4), c(2,3),c(4,4))

md1 = 0.2
md2 = 0.1

comp1 <- create_component(v1, md1, type="POINT")
comp2 <- create_component(v2, md2, type="POINT")

# Creating two components of the type LINE

md3 = 0.45
md4 = 0.32

v3 = rbind(c(2,2), c(3,3))
v4 = rbind(c(1,1), c(3,2))

comp3 <- create_component(v3, md3, type="LINE")
comp4 <- create_component(v4, md4, type="LINE")

# Creating two components of the type REGION

p1 <- rbind(c(0,0), c(1,0), c(3,2), c(2,4), c(1,4), c(0,0))
p2 <- rbind(c(1,1), c(1,2), c(2,2), c(1,1))
list_pols_1 <- list(p1,p2)

p3 <- rbind(c(1,0), c(2,0), c(4,2), c(3,4), c(2,4), c(1,0))
p4 <- rbind(c(2,2), c(2,3), c(3,4), c(2,2))
list_pols_2 <- list(p3,p4)

comp_pol1 <- create_component(list_pols_1, 0.4, "REGION")
comp_pol2 <- create_component(list_pols_2, 0.6, "REGION")


# Creating components with an sfg object
library(sf)

# POINT
md1 <- 0.2
pts1 <- rbind(c(1, 2), c(3, 2))
comp1 <- component_from_sfg(st_multipoint(pts1), md1) 

# LINE
md2 <- 0.1
pts2 <- rbind(c(2, 2), c(3, 3))
comp2 <- component_from_sfg(st_linestring(pts2), md2) 

# REGION
md3 <- 0.4
matrix_object = matrix(c(1,1,8,1,8,8,1,8,1,1),ncol=2, byrow=TRUE)
pts3 = list(matrix_object)
comp3 = component_from_sfg(st_polygon(pts3), md3)

}
