test_that("can reorder by 2d summary", {
  df <- tibble::tribble(
    ~g,  ~x,
    "a", 3,
    "a", 3,
    "b", 2,
    "b", 2,
    "b", 1
  )

  f1 <- fct_reorder(df$g, df$x)
  expect_equal(levels(f1), c("b", "a"))

  f2 <- fct_reorder(df$g, df$x, .desc = TRUE)
  expect_equal(levels(f2), c("a", "b"))
})

test_that("can reorder by 2d summary", {
  df <- tibble::tribble(
    ~g, ~x, ~y,
    "a", 1, 10,
    "a", 2, 5,
    "b", 1, 5,
    "b", 2, 10
  )

  f1 <- fct_reorder2(df$g, df$x, df$y)
  expect_equal(levels(f1), c("b", "a"))

  f2 <- fct_reorder(df$g, df$x, .desc = TRUE)
  expect_equal(levels(f2), c("a", "b"))
})

test_that("missing groups appear at end", {
  df <- tibble::tribble(
    ~g, ~x, ~y,
    "a", NA, NA,
    "b", 1, 10,
  )

  f1 <- fct_reorder2(df$g, df$x, df$y)
  expect_equal(levels(f1), c("b", "a"))
})

test_that("first2/last2 return expected values", {
  expect_equal(first2(4:1, 1:4), 4)
  expect_equal(last2(4:1, 1:4), 1)
})

test_that("last2 ignores points where either value is missing", {
  expect_equal(last2(1:4, c(1:3, NA)), 3)
  expect_equal(last2(c(1:3, NA), 1:4), 3)
})

test_that("last2 returns NA if no non-missing pairs", {
  expect_equal(last2(c(NA, 1), c(1, NA)), NA_real_)
  expect_equal(last2(c(NA, 1), c("x", NA)), NA_character_)
})

test_that("complains if summary doesn't return single value", {
  fun1 <- function(x, y) c(1, 2)
  fun2 <- function(x, y) integer()

  expect_snapshot(error = TRUE, {
    fct_reorder("a", 1, fun1)
    fct_reorder("a", 1, fun2)
    fct_reorder2("a", 1, 2, fun1)
    fct_reorder2("a", 1, 2, fun2)
  })
})

test_that("fct_infreq respects missing values", {
  f <- factor(c("a", "b", "b", NA, NA, NA), exclude = FALSE)
  expect_equal(levels(fct_infreq(f)), c(NA, "b", "a"))
})

test_that("fct_inseq sorts in numeric order", {
  x <- c("1", "2", "3")
  expect_equal(
    fct_inseq(factor(x, levels = c("3", "1", "2"))),
    factor(x, levels = c("1", "2", "3"))
  )

  # non-numeric go to end
  x <- c("1", "2", "3", "a")
  expect_equal(
    fct_inseq(factor(x, levels = c("a", "3", "1", "2"))),
    factor(x, levels = c("1", "2", "3", "a"))
  )
})

test_that("fct_inseq gives error for non-numeric levels", {
  f <- factor(c("c", "a", "a", "b"))
  expect_error(levels(fct_inseq(f)), "level must be coercible to numeric")
})
