#' Read an ATS Acoustic Receiver File
#'
#' This function takes a raw acoustic detection file generated by an ATS JSATS
#' receiver and processes it into a dataframe which can be used by the filtering
#' functions in this package.This is called within read_jsats().
#'
#' @param path the path to the folder containing the desired file
#' @param file the path of the desired file
#' @param timezone the Olsen Named time zone, default is "America/Los_Angeles"
#' @returns A dataframe converting the raw detection data into rows of detections
#' @export
#' @examples
#' # Read in an ATS file
#' path = system.file("extdata/", package = "filteRjsats")
#' read_ats(path = path, file = "17111___210531_115356.csv",
#' timezone = "America/Los_Angeles")
#' # Warnings are expected due to the formatting of ATS files
read_ats <- function(path, file, timezone="America/Los_Angeles"){
  ATS <- data.frame(read.delim(file = file.path(path,file),
                               skip = 0))
  colnames(ATS) <- "lines"
  ATS$G72 = grepl("G72",ATS$lines)
  ATS$commas = lengths(regmatches(ATS$lines, gregexpr(",",ATS$lines)))
  ATS$valid = ifelse(ATS$G72 == TRUE & ATS$commas == 13, TRUE, FALSE)
  ATS <- ATS[ATS$valid == TRUE,]
  ATS <- tidyr::separate(data = ATS,
                         col = lines,
                         sep = ",",
                         into = c(NA,NA,NA,NA,'DateTime','TagCode',
                                  'Tilt', 'VBatt', 'Temp', 'Pressure',
                                  'SigStr', 'Bit_Period', 'Threshold'))
  ATS$DT_Check = grepl('\\.',ATS$DateTime)
  ATS <- ATS[ATS$DT_Check == TRUE,]
  ATS$DateTime_Local = lubridate::mdy_hms(ATS$DateTime, tz = timezone)
  ATS$FullID = ATS$TagCode
  ATS$Tag_Hex = as.character(substr(ATS$TagCode,5,8)) #There's a space before the G
  ATS$Tag_Decimal = broman::hex2dec(ATS$Tag_Hex)
  ATS$Tilt = as.numeric(ATS$Tilt)
  ATS$Volt = as.numeric(ATS$VBatt)
  ATS$Temp = as.numeric(ATS$Temp)
  ATS$SigStr = as.numeric(ATS$SigStr)
  ATS$BitPer = ATS$Bit_Period
  ATS <- tidyr::separate(data = ATS,
                         col = BitPer,
                         into = c(NA,"B1","B2"),
                         sep = "\\s")
  ATS <- tidyr::separate(data = ATS,
                         col = B2,
                         into = c("B3","B4"),
                         sep = "/")
  ATS$Freq = ifelse(is.na(ATS$B3),-999, 100000/(as.numeric(ATS$B1)+(as.numeric(ATS$B3)/31)))
  ATS$Thres = ifelse(is.na(ATS$B3), -999, as.numeric(ATS$Threshold))
  ReceiverSN <-  gsub("SR","",stringr::str_split(file, pattern = '\\.')[[1]][1])
  ReceiverSN <- stringr::str_split(ReceiverSN,pattern = '_')[[1]][1]
  ATS$ReceiverSN <- rep(ReceiverSN, length(ATS$DateTime_Local))
  ATS$Make <- rep("ATS", length(ATS$DateTime_Local))
  ATS <- dplyr::select(.data =  ATS, ReceiverSN, Make, DateTime_Local,
                       Tag_Decimal, Tag_Hex, Tilt, Volt, Temp, SigStr, Freq,
                       Thres)
  ATS
}
