% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FEMevaluator.R
\name{eval.FEM}
\alias{eval.FEM}
\title{Evaluate a FEM object at a set of point locations}
\usage{
eval.FEM(FEM, locations = NULL, incidence_matrix = NULL, search = "tree", 
                bary.locations = NULL)
}
\arguments{
\item{FEM}{A \code{FEM} object to be evaluated.}

\item{locations}{A 2-columns (in 1.5D or 2D) or 3-columns (in 2.5D and 3D) matrix with the spatial locations where the
FEM object should be evaluated.}

\item{incidence_matrix}{In case of areal evaluations, the #regions-by-#elements incidence matrix defining the regions
where the FEM object should be evaluated.}

\item{search}{a flag to decide the search algorithm type (tree or naive or walking search algorithm).}

\item{bary.locations}{A list with three vectors:
\code{locations}, location points which are same as the given locations options. (checks whether both locations are the same);
\code{element ids}, a vector of element id of the points from the mesh where they are located;
\code{barycenters}, a vector of barycenter of points from the located element.}
}
\value{
A vector or a matrix of numeric evaluations of the \code{FEM} object.
If the \code{FEM} object contains multiple finite element functions the output is a matrix, and
each row corresponds to the location (or areal region) where the evaluation has been taken, while each column
corresponds to the function evaluated.
}
\description{
It evaluates a FEM object at the specified set of locations or areal regions. The locations are used for
pointwise evaluations and incidence matrix for areal evaluations.
The locations and the incidence matrix cannot be both NULL or both provided.
}
\examples{
library(fdaPDE)
## Upload the horseshoe2D data
data(horseshoe2D)
boundary_nodes = horseshoe2D$boundary_nodes
boundary_segments = horseshoe2D$boundary_segments
locations = horseshoe2D$locations

## Create the 2D mesh
mesh = create.mesh.2D(nodes = rbind(boundary_nodes, locations), segments = boundary_segments)
## Create the FEM basis
FEMbasis = create.FEM.basis(mesh)
## Compute the coeff vector evaluating the desired function at the mesh nodes
## In this case we consider the fs.test() function introduced by Wood et al. 2008
coeff = fs.test(mesh$nodes[,1], mesh$nodes[,2])
## Create the FEM object
FEMfunction = FEM(coeff, FEMbasis)

## Evaluate the finite element function in the location (1,0.5)
eval.FEM(FEMfunction, locations = matrix(c(1, 0.5), ncol = 2))

## Evaluate the mean of the finite element function over the fifth triangle of the mesh
incidence_matrix = matrix(0, ncol = nrow(mesh$triangles))
incidence_matrix[1,5] = 1
eval.FEM(FEMfunction, incidence_matrix = incidence_matrix)
}
\references{
\itemize{
   \item{Sangalli, L. M., Ramsay, J. O., & Ramsay, T. O. (2013). Spatial spline regression models.
Journal of the Royal Statistical Society: Series B (Statistical Methodology), 75(4), 681-703.}
   \item{Azzimonti, L., Sangalli, L. M., Secchi, P., Domanin, M., & Nobile, F. (2015). Blood flow velocity field estimation
via spatial regression with PDE penalization. Journal of the American Statistical Association, 110(511), 1057-1071.}
}
}
