% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fdaPDE.smoothing.R
\name{smooth.FEM.PDE.sv.basis}
\alias{smooth.FEM.PDE.sv.basis}
\title{Spatial regression with differential regularization: anysotropic and non-stationary case (elliptic PDE with space-varying coefficients)}
\usage{
smooth.FEM.PDE.sv.basis(locations = NULL, observations, FEMbasis, 
 lambda, PDE_parameters, covariates = NULL, BC = NULL, GCV = FALSE, 
 CPP_CODE = TRUE)
}
\arguments{
\item{locations}{A #observations-by-2 matrix where each row specifies the spatial coordinates \code{x} and \code{y} of the corresponding observations in the vector \code{observations}.
This parameter can be \code{NULL}. In this case the spatial coordinates of the corresponding observations are assigned as specified in \code{observations}.}

\item{observations}{A vector of length #observations with the observed data values over the domain. 
The locations of the observations can be specified with the \code{locations} argument. 
Otherwise if only the vector of observations is given, these are consider to be located in the corresponding node in the table
\code{nodes} of the mesh. In this last case, an \code{NA} value in the \code{observations} vector indicates that there is no observation associated to the corresponding
 node.}

\item{FEMbasis}{A \code{FEMbasis} object describing the Finite Element basis, as created by \code{\link{create.FEM.basis}}.}

\item{lambda}{A scalar or vector of smoothing parameters.}

\item{PDE_parameters}{A list specifying the space-varying parameters of the elliptic PDE in the regularizing term: \code{K}, a function that for each spatial location in the spatial domain 
(indicated by the vector of the 2 spatial coordinates) returns a 2-by-2 matrix of diffusion coefficients. This induces an anisotropic 
smoothing with a local preferential direction that corresponds to the first eigenvector of the diffusion matrix K.The function must support recycling for efficiency reasons, thus if the input parameter is a #point-by-2 matrix, the output should be
an array with dimensions 2-by-2-by-#points.\code{b}, a function that for each spatial location in the spatial domain returns 
a vector of length 2 of transport coefficients. This induces a local smoothing only in the direction specified by the vector \code{b}. The function must support recycling for efficiency reasons, thus if the input parameter is a #point-by-2 matrix, the output should be
a matrix with dimensions 2-by-#points; \code{c}, a function that for each spatial location in the spatial domain  returns a scalar reaction coefficient.
\code{c} induces a shrinkage of the surface to zero. The function must support recycling for efficiency reasons, thus if the input parameter is a #point-by-2 matrix, the output should be
a vector with length #points; \code{u}, a function that for each spatial location in the spatial domain  returns a scalar reaction coefficient.
\code{u} induces a reaction effect. The function must support recycling for efficiency reasons, thus if the input parameter is a #point-by-2 matrix, the output should be
a vector with length #points.}

\item{covariates}{A #observations-by-#covariates matrix where each row represents the covariates associated with the corresponding observed data value in \code{observations}.}

\item{BC}{A list with two vectors: 
\code{BC_indices}, a vector with the indices in \code{nodes} of boundary nodes where a Dirichlet Boundary Condition should be applied;
\code{BC_values}, a vector with the values that the spatial field must take at the nodes indicated in \code{BC_indices}.}

\item{GCV}{Boolean. If \code{TRUE} the following quantities are computed: the trace of the smoothing matrix, the estimated error standard deviation,  and 
the Generalized Cross Validation criterion, for each value of the smoothing parameter specified in \code{lambda}.}

\item{CPP_CODE}{Boolean. If \code{TRUE} the computation relies on the C++ implementation of the algorithm. This usually ensures a much faster computation.}
}
\value{
A list with the following variables:
         \item{\code{fit.FEM}}{A \code{FEM} object that represents the fitted spatial field.}
         \item{\code{PDEmisfit.FEM}}{A \code{FEM} object that represents the PDE misfit for the estimated spatial field.}
         \item{\code{beta}}{If covariates is not \code{NULL}, a matrix with number of rows equal to the number of covariates and numer of columns equal to length of lambda.  The \code{j}th column represents the vector of regression coefficients when 
         the smoothing parameter is equal to \code{lambda[j]}.}
         \item{\code{edf}}{If GCV is \code{TRUE}, a scalar or vector with the trace of the smoothing matrix for each value of the smoothing parameter specified in \code{lambda}.}
         \item{\code{stderr}}{If GCV is \code{TRUE}, a scalar or vector with the estimate of the standard deviation of the error for each value of the smoothing parameter specified in \code{lambda}.}
         \item{\code{GCV}}{If GCV is \code{TRUE}, a  scalar or vector with the value of the GCV criterion for each value of the smoothing parameter specified in \code{lambda}.}
}
\description{
This function implements a spatial regression model with differential regularization; anysotropic and non-stationary case. In particular, the regularizing term involves a second order elliptic PDE with space-varying coefficients, that models the space-variation of the phenomenon. Space-varying covariates can be included in the model. The technique accurately handle data distributed over irregularly shaped domains. Moreover, various conditions can be imposed at the domain boundaries.
}
\examples{
# Loading the mesh
data(mesh.2D.rectangular)
# Create the FEM basis object
FEMbasis = create.FEM.basis(mesh.2D.rectangular)
# Create a vector of noisy samples of an underlying spatial field, 
# located over the nodes of the mesh
observations = sin(0.2*pi*mesh.2D.rectangular$nodes[,1]) + 
rnorm(n = nrow(mesh.2D.rectangular$nodes), sd = 0.1)
# Set the smoothing coefficient
lambda = c(10^-2)
#Set the space vriant coefficients of the penalizying PDE
K_func<-function(points)
{
mat<-c(0.01,0,0,1)
output = array(0, c(2, 2, nrow(points)))
for (i in 1:nrow(points))
   output[,,i] = 0.5*mat \%*\% t(points[i,1]^2)
output
}
b_func<-function(points)
{
output = array(0, c(2, nrow(points)))
for (i in 1:nrow(points))
   output[,i] = 0
output
}

c_func<-function(points)
{
rep(c(0), nrow(points))
}

u_func<-function(points)
{
rep(c(0), nrow(points))
}
# Assemble the parameters in one object
PDE_parameters = list(K = K_func, b = b_func, c = c_func, u = u_func)
# Estimate the underlying spatial field and plot these
FEM_CPP_PDE = smooth.FEM.PDE.sv.basis(observations = observations, 
             FEMbasis = FEMbasis, lambda = lambda, PDE_parameters = PDE_parameters)
plot(FEM_CPP_PDE$fit.FEM)
}
\references{
Azzimonti, L., Sangalli, L.M., Secchi, P., Domanin, M., and Nobile, F., 2014. Blood flow velocity field estimation via spatial regression with PDE penalization Blood flow velocity field estimation via spatial regression with PDE penalization. DOI. 10.1080/01621459.2014.946036. \cr
 Azzimonti, L., Nobile, F., Sangalli, L.M., and Secchi, P., 2014. Mixed Finite Elements for Spatial Regression with PDE Penalization. SIAM/ASA Journal on Uncertainty Quantification, 2(1), pp.305-335.
}
\seealso{
\code{\link{smooth.FEM.basis}}, \code{\link{smooth.FEM.PDE.basis}}
}

