library(testthat)

test_that("editmatrix works correcly with character",{
   cond <- c( "x == y"
            , "z + w == y + x"
			   , "x + z == y + 2*w"
			   )
			
   mat <- editmatrix(editrules=cond)
   expect_equal(as.integer(mat[1,]), c(1,-1,0,0))
   expect_equal(as.integer(mat[2,]), c(-1,-1,1,1))
   expect_equal(as.integer(mat[3,]), c(1,-1,-2,1))
})

test_that("editmatrix can simplify",{
   cond <- c( "2*x == x + y"
            , "z + 2 == y + 3"
            , "w == 3"
			   )
			
   mat <- editmatrix(editrules=cond)
   C <- getC(mat)
   expect_equal(as.integer(mat[1,]), c(1,-1,0,0))
   expect_equal(as.integer(mat[2,]), c(0,-1,1,0))
   expect_equal(C[2], c(e2=1))
   
   expect_equal(mat[3,], c(x=0,y=0,z=0,w=1))
   expect_equal(C[3], c(e3=3))
})


test_that("editmatrix works correcly with data.frame",{
   
   edtinf.csv <- 
"name,edit
A,x == y
B,z + w == y + x
C,z == y + 2*w
"
   edtinf <- read.csv(textConnection(edtinf.csv))
			
   mat <- editmatrix(edtinf)
   expect_equal(as.integer(mat[1,]), c(1,-1,0,0))
   expect_equal(as.integer(mat[2,]), c(-1,-1,1,1))
   expect_equal(as.integer(mat[3,]), c(0,-1,-2,1))
})

test_that("editmatrix works with constants",{
   cond <- c( "x + y > 2"
            , "y < 10"
            )
   mat <- editmatrix(cond)
   expect_equal(as.integer(mat[1,]), c(1,1))
   expect_equal(as.integer(mat[2,]), c(0,1))
   expect_equal(as.integer(getC(mat)), c(2,10))
})

test_that("conditional statement parsing is not working..",{
   expect_error(editmatrix("if(x < 2) y > 4"))
})

test_that("editmatrix works with negative constants",{
   cond <- c( "x + y > -2"
            , "y < -10"
            )
   mat <- editmatrix(cond)
   expect_equal(as.integer(mat[1,]), c(1,1))
   expect_equal(as.integer(mat[2,]), c(0,1))
   expect_equal(as.integer(getC(mat)), c(-2,-10))
})

test_that("editmatrix works with negative coefficients",{
   cond <- c( "-2*x + y > 2"
            )
   mat <- editmatrix(editrules=cond)
   expect_equivalent(mat[1,], c(-2,1))
   expect_equivalent(getC(mat), c(2))
})

test_that("editmatrix fails with nonconstant coefficient",{
   cond <- c( "a*x == 2"
            )
   mat <- editmatrix(editrules=cond)
})


test_that("is.editmatrix works",{
   mat <- editmatrix(editrules="x==y")
   expect_true(is.editmatrix(mat))
   expect_false(is.editmatrix(unclass(mat)))
})

test_that("editrules returns attribute",{
   mat <- editmatrix(editrules="x==y")
   expect_equal(editrules(mat), attr(mat, "editrules"))
})


test_that("as.editmatrix works",{
   mat <- matrix( c( 1,-2, 0
                   , 2, 0, 1
				   )
				, nrow=2
				, byrow=TRUE
	#			, dimnames=list(c("a", "b"), c("x","y", "z"))
				)
   E <- as.editmatrix(mat, C=c(0,1), ops=c("==","<"))
   ei <- editrules(E)
   expect_equal(ei$edit, c("a == 2*b", "2*a + c < 1"))
})

test_that("editmatrix normalize works",{
   cond <- c( "x > y"
            , "z + w >= y + x"
			   , "x + z < y + 2*w"
			   , "x + z == y + 2*w"
			   , "x + z <= y + 2*w"
			   )
			
   mat <- editmatrix(editrules=cond, normalize=TRUE)
   print(mat)
   
   expect_equivalent(mat[1,], c(-1,1,0,0))
   expect_equivalent(mat[2,], c(1,1,-1,-1))
   expect_equivalent(mat[3,], c(1,-1,-2,1))
   expect_equivalent(mat[4,], c(1,-1,-2,1))
   expect_equivalent(mat[5,], c(1,-1,-2,1))
   ops <- getOps(mat)
   expect_equivalent(ops, c("<=", "<", "<","==", "<="))
})