% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/define_retra.R
\name{define_retra}
\alias{define_retra}
\title{Define representative trajectories from trajectory features}
\usage{
define_retra(data, d = NULL, trajectories = NULL, states = NULL, retra = NULL)
}
\arguments{
\item{data}{A data frame of four columns indicating identifiers for the new
representative trajectories, the individual trajectories or sites to which the
states belong, the order of the states in the individual trajectories, and the
identifier of the representative trajectory to which the states belong (only
if \code{!is.null(retra)}). Alternatively, 'data' can be a vector or a list of
character vectors including the sequence of segments forming the new
representative trajectory. See Details for further clarifications to define
\code{data}.}

\item{d}{Either a symmetric matrix or an object of class \code{\link{dist}} containing the
dissimilarities between each pair of states of all trajectories in the EDR.
If \code{NULL} (default), the length (\code{Length}) of the new representative trajectories
and the distances between states of different trajectories or sites (\code{Link_distance})
are not calculated.}

\item{trajectories}{Only needed if \code{!is.null(d)}. Vector indicating the
trajectory or site to which each state in \code{d} belongs.}

\item{states}{Only needed if \code{!is.null(d)}. Vector of integers indicating the
order of the states in \code{d} for each trajectory.}

\item{retra}{Object of class \code{RETRA} returned from \code{\link[=retra_edr]{retra_edr()}}. If \code{NULL}
(default), \code{minSegs} and \code{Seg_density} are not provided for the new representative
trajectories.}
}
\value{
An object of class \code{RETRA}, which is a list of length equal to the number of
representative trajectories defined. For each trajectory, the following
information is returned:
\describe{
\item{\code{minSegs}}{Value of the \code{minSegs} parameter used in \code{\link[=retra_edr]{retra_edr()}}. If \code{retra}
is \code{NULL}, \code{minSegs} = \code{NA}.}
\item{\code{Segments}}{Vector of strings including the sequence of segments forming the
representative trajectory. Each segment is identified by a string of the form
\code{traj[st1-st2]}, where \code{traj} is the identifier of the original trajectory to
which the segment belongs and \code{st1} and \code{st2} are identifiers of the initial
and final states defining the segment. The same format \code{traj[st1-st2]} is
maintained when only one state of an individual trajectory is considered
(\code{st1} = \code{st2}). \code{traj}, \code{st1}, and \code{st2} are recycled from \code{data}.}
\item{\code{Size}}{Integer indicating the number of states forming the representative
trajectory.}
\item{\code{Length}}{Numeric value indicating the length of the representative trajectory,
calculated as the sum of the dissimilarities in \code{d} between every pair of
consecutive states. If \code{d} is \code{NULL}, \code{Length} = \code{NA}.}
\item{\code{Link_distance}}{Data frame of two columns indicating artificial links between
two segments (\code{Link}) and the dissimilarity between the connected states
(\code{Distance}). When two representative segments are linked by a common state or
by two consecutive states of the same trajectory, the link distance is zero or
equal to the length of a real segment, respectively. In both cases, the link
is not considered in the returned data frame. If \code{d} is \code{NULL}, \code{Link_distance}
= \code{NA}.}
\item{\code{Seg_density}}{Data frame of two columns and one row for each representative
segment. \code{Density} contains the number of segments in the EDR that is represented
by each segment of the representative trajectory. \code{kdTree_depth} contains the
depth of the k-d tree for each leaf represented by the corresponding segment.
That is, the number of partitions of the ordination space until finding a region
with \code{minSegs} segments or less. If \code{retra} is \code{NULL}, \code{Seg_density} = \code{NA}.}
}
}
\description{
Generate an object of class \code{RETRA} from a data frame containing trajectory
states to define representative trajectories in Ecological Dynamic Regimes (EDR).
}
\details{
Each representative trajectory returned by the function \code{\link[=retra_edr]{retra_edr()}} corresponds
to the longest sequence of representative segments that can be linked according
to the criteria defined in the RETRA-EDR algorithm (Sánchez-Pinillos et al.,
2023). One could be interested in splitting the obtained trajectories, considering
only a fraction of the returned trajectories, or defining representative
trajectories following different criteria than those in RETRA-EDR.
The function \code{define_retra()} allows generating an object of class \code{RETRA} that
can be used in other functions of \code{ecoregime} (e.g., \code{\link[=plot]{plot()}}).

For that, it is necessary to provide information about the set of segments or
trajectory states that form the new representative trajectory through the
argument \code{data}:
\itemize{
\item \code{data} can be defined as a \strong{data frame} with as many rows as the number of
states in all representative trajectories and the following columns:
\describe{
\item{\code{RT}}{A string indicating the identifier of the new representative trajectories.
Each identifier needs to appear as many times as the number of states forming
each representative trajectory.}
\item{\code{RT_traj}}{A vector indicating the individual trajectories in the EDR to which
each state of the new representative trajectory belongs.}
\item{\code{RT_states}}{A vector of integers indicating the identifier of the states
forming the new representative trajectories. Each integer must refer to the order
of the states in the individual trajectories of the EDR to which they belong.}
\item{\code{RT_retra}}{Only if the new trajectories are defined from representative
trajectories returned by \code{\link[=retra_edr]{retra_edr()}} (when \code{!is.null(retra)}). A vector of strings
indicating the representative trajectory in \code{retra} to which each state belongs.}
}
\item Alternatively, \code{data} can be defined as either a \strong{vector} (if there is one
representative trajectory) or a \strong{list of character vectors} (with as many
elements as the number of representative trajectories desired) containing the
sequence of segments of the representative trajectories. In any case, each
segment needs to be specified in the form \code{traj[st1-st2]}, where \code{traj} is the
identifier of the original trajectory to which the segment belongs and \code{st1}
and \code{st2} are identifiers of the initial and final states defining the segment.
If only one state of an individual trajectory is considered to form the
representative trajectory, the corresponding segment needs to be defined as
\code{traj[st-st]}.
}
}
\examples{
# Example 1 -----------------------------------------------------------------
# Define representative trajectories from the outputs of retra_edr().

# Identify representative trajectories using retra_edr()
d <- EDR_data$EDR1$state_dissim
trajectories <- EDR_data$EDR1$abundance$traj
states <- EDR_data$EDR1$abundance$state
old_retra <- retra_edr(d = d, trajectories = trajectories, states = states,
                       minSegs = 5)

# retra_edr() returns three representative trajectories
old_retra

# Keep the last five segments of trajectories "T2" and "T3"
selected_segs <- old_retra$T2$Segments[4:length(old_retra$T2$Segments)]

# Identify the individual trajectories for each state...
selected_segs
selected_traj <- rep(c(15, 4, 4, 1, 14), each = 2)

# ...and the states (in the same order as the representative trajectory).
selected_states <- c(1, 2, 2, 3, 3, 4, 1, 2, 2, 3)

# Generate the data frame with the format indicated in the documentation
df <- data.frame(RT = rep("A", length(selected_states)),
                 RT_traj = selected_traj,
                 RT_states = as.integer(selected_states),
                 RT_retra = rep("T2", length(selected_states)))

# Remove duplicates (trajectory 4, state 3)
df <- unique(df)

# Generate a RETRA object using define_retra()
new_retra <- define_retra(data = df,
                          d = d,
                          trajectories = trajectories,
                          states = states,
                          retra = old_retra)

# Example 2 -----------------------------------------------------------------
# Define representative trajectories from sequences of segments

# Select all segments in T1, split T2 into two new trajectories, and include
# a trajectory composed of states belonging to trajectories "5", "6", and "7"
data <- list(old_retra$T1$Segments,
             old_retra$T2$Segments[1:3],
             old_retra$T2$Segments[4:8],
             c("5[1-2]", "5[2-3]", "7[4-4]", "6[4-5]"))

# Generate a RETRA object using define_retra()
new_retra <- define_retra(data = data,
                          d = d,
                          trajectories = trajectories,
                          states = states,
                          retra = old_retra)

# Example 3 -----------------------------------------------------------------
# Define two representative trajectories from individual trajectories in EDR1.

# Define trajectory "A" from states in trajectories 3 and 4
data_A <- data.frame(RT = rep("A", 4),
                     RT_traj = c(3, 3, 4, 4),
                     RT_states = c(1:2, 4:5))

# Define trajectory "B" from states in trajectories 5, 6, and 7
data_B <- data.frame(RT = rep("B", 5),
                     RT_traj = c(5, 5, 7, 6, 6),
                     RT_states = c(1, 2, 4, 4, 5))

# Compile data for both trajectories in a data frame
df <- rbind(data_A, data_B)
df$RT_states <- as.integer(df$RT_states)

# Generate a RETRA object using define_retra()
new_retra <- define_retra(data = df, d = EDR_data$EDR1$state_dissim,
                          trajectories = EDR_data$EDR1$abundance$traj,
                          states = EDR_data$EDR1$abundance$state)


}
\seealso{
\code{\link[=retra_edr]{retra_edr()}} for identifying representative trajectories in EDRs through
RETRA-EDR.

\code{\link[=summary]{summary()}} for summarizing the characteristics of the representative
trajectories.

\code{\link[=plot]{plot()}} for plotting representative trajectories in an ordination space
representing the state space of the EDR.
}
\author{
Martina Sánchez-Pinillos
}
