% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/number_line.R
\name{number_line}
\alias{number_line}
\alias{as.number_line}
\alias{is.number_line}
\alias{left_point}
\alias{right_point}
\alias{start_point}
\alias{end_point}
\alias{number_line_width}
\alias{reverse_number_line}
\alias{shift_number_line}
\alias{expand_number_line}
\alias{compress_number_line}
\alias{number_line_sequence}
\title{Number line objects}
\usage{
number_line(l, r, id = NULL, gid = NULL)

as.number_line(x)

is.number_line(x)

left_point(x)

right_point(x)

start_point(x)

end_point(x)

number_line_width(x)

reverse_number_line(x, direction = "both")

shift_number_line(x, by = 1)

expand_number_line(x, by = 1, point = "both")

compress_number_line(x, method = c("across", "chain", "aligns_start",
  "aligns_end", "inbetween"), collapse = FALSE, deduplicate = TRUE)

number_line_sequence(x, by = 1)
}
\arguments{
\item{l}{Left point of the \code{number_line} object. Should be, or can be coerced to a \code{numeric} object}

\item{r}{Right point of the \code{number_line} object. Should be, or can be coerced to a \code{numeric} object}

\item{id}{Unique \code{numeric} ID. Providing this is optional}

\item{gid}{Unique \code{numeric} Group ID. Providing this is optional}

\item{x}{\code{number_line} object}

\item{direction}{Type of \code{"number_line"} objects whose direction are to be reversed. Options are; \code{"increasing"}, \code{"decreasing"} or \code{"both"}.}

\item{by}{increment or decrement}

\item{point}{\code{"start"} or \code{"start"} point}

\item{method}{Method of overlap}

\item{collapse}{If \code{TRUE}, collapse the compressed results based on \code{method} of overlaps}

\item{deduplicate}{if \code{TRUE}, retains only one \code{number_line} object among duplicates}
}
\value{
\code{number_line} object
}
\description{
A set of functions to create and manipulate \code{number_line} objects.
}
\details{
A \code{number_line} object represents a series of real numbers on a number line.

Visually, it's presented as the left (\code{l}) and right (\code{r}) points of the series. This may differ from start and end points.
The start point is the lowest number in the series, regardless of whether it's at the left or right point..

The location of the start point - left or right, indicate if it's an \code{"increasing"} or \code{"decreasing"} series.
This is referred to as the \code{direction} of the \code{number_line} object.

\code{reverse_number_line()} - reverses the direction of a \code{number_line} object. A reversed \code{number_line} object has its \code{l} and \code{r} points swapped but maintains the same width or length.
The \code{direction} argument determines which type of \code{number_line} objects will be reversed.
\code{number_line} objects with non-finite numeric starts or end points i.e. (\code{NA}, \code{NaN} and \code{Inf}) can't be reversed.

\code{shift_number_line()} - a convenience function to shift a \code{number_line} object towards the positive or negative end of the number line.

\code{expand_number_line()} - a convenience function to increase or decrease the width or length of a \code{number_line} object.

\code{compress_number_line()} - Collapses overlapping \code{number_line} objects into a new \code{number_line} objects that covers the start and end points of the originals.
This results in duplicate \code{number_line} objects with start and end points of the new expanded \code{number_line} object.
See \code{\link{overlap}} for further details on overlapping \code{number_line} objects.
If a familiar (but unique) \code{id} is used when creating the \code{number_line} objects,
\code{compress_number_line()} can be a simple alternative to \code{\link{record_group}} or \code{\link{episode_group}}.

\code{number_line_sequence()} - a convenience function to convert a \code{number_line} object into a sequence of finite numbers. The sequence will also include the start and end points.
The direction of the sequence will correspond to that of the \code{number_line} object.
}
\examples{
library(lubridate)

number_line(-100, 100); number_line(10, 11.2)

# Other numeric based object classes are also compatible for numeric_line objects
number_line(dmy_hms("15/05/2019 13:15:07"), dmy_hms("15/05/2019 15:17:10"))

# A warning is given if 'l' and 'r' have different classes. Consider if these need to be corrected
number_line(2, dmy("05/01/2019"))

# Convert numeric based objects to number_line objects
as.number_line(5.1); as.number_line(dmy("21/10/2019"))

# Test for number_line objects
a <- number_line(0, -100)
b <- number_line(dmy("25/04/2019"), dmy("01/01/2019"))
is.number_line(a); is.number_line(b)

# Structure of a number_line object
left_point(a); right_point(a); start_point(a); end_point(a)

# Reverse number_line objects
reverse_number_line(number_line(dmy("25/04/2019"), dmy("01/01/2019")))
reverse_number_line(number_line(200,-100), "increasing")
reverse_number_line(number_line(200,-100), "decreasing")

# Shift number_line objects
number_line(5,6)
# Towards the positive end of the number line
shift_number_line(number_line(5,6), 2)
# Towards the negative end of the number line
shift_number_line(number_line(6,1), -2)

# Increase or reduce the width or length of a \\code{number_line} object
c(number_line(3,6), number_line(6,3))
expand_number_line(c(number_line(3,6), number_line(6,3)), 2)
expand_number_line(c(number_line(3,6), number_line(6,3)), -1)
expand_number_line(c(number_line(3,6), number_line(6,3)), 2, "start")
expand_number_line(c(number_line(3,6), number_line(6,3)), -2, "end")

# Collapse number line objects
x <- c(number_line(10,10), number_line(10,20), number_line(5,30),  number_line(30,40))
compress_number_line(x, deduplicate = FALSE)
compress_number_line(x)
compress_number_line(x, collapse=TRUE)
compress_number_line(x, collapse=TRUE, method = "inbetween")

# Convert a number line object to its series of real numbers
number_line_sequence(number_line(1, 5))
number_line_sequence(number_line(5, 1), .5)
number_line_sequence(number_line(dmy("01/04/2019"), dmy("10/04/2019")), 1)

# The length of the resulting vector will depend on the object class
number_line_sequence(number_line(dmy("01/04/2019"), dmy("04/04/2019")), 1.5)

nl <- number_line(dmy_hms("01/04/2019 00:00:00"), dmy_hms("04/04/2019 00:00:00"))
head(number_line_sequence(nl, 1.5), 15)
d <- duration(1.5,"days")
number_line_sequence(nl, d)

}
