% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/validate.R
\name{validate}
\alias{validate}
\title{Validate regression models on a test set}
\usage{
validate(train_data, models, test_data = NULL,
  partitions_col = ".partitions", family = "gaussian", link = NULL,
  control = NULL, REML = FALSE, cutoff = 0.5, positive = 2,
  metrics = list(), err_nc = FALSE, rm_nc = FALSE,
  parallel = FALSE, model_verbose = FALSE)
}
\arguments{
\item{train_data}{Data Frame.}

\item{models}{Model formulas as strings. (Character)

 E.g. \code{c("y~x", "y~z")}.

 Can contain random effects.

 E.g. \code{c("y~x+(1|r)", "y~z+(1|r)")}.}

\item{test_data}{Data Frame. If specifying \code{partitions_col}, this can be \code{NULL}.}

\item{partitions_col}{Name of grouping factor for identifying partitions. (Character)

 Rows with the value \code{1} in \code{partitions_col} are used as training set and
 rows with the value \code{2} are used as test set.

 N.B. Only used if \code{test_data} is \code{NULL}.}

\item{family}{Name of family. (Character)

 Currently supports \code{"gaussian"} and \code{"binomial"}.}

\item{link}{Link function. (Character)

 E.g. \code{link = "log"} with \code{family = "gaussian"} will
 use \code{family = gaussian(link = "log")}.

 See \code{\link[stats:family]{stats::family}} for available link functions.

 \subsection{Default link functions}{

 Gaussian: \code{'identity'}.

 Binomial: \code{'logit'}.}}

\item{control}{Construct control structures for mixed model fitting
 (i.e. \code{\link[lme4]{lmer}} and \code{\link[lme4]{glmer}}).
 See \code{\link[lme4:lmerControl]{lme4::lmerControl}} and
 \code{\link[lme4:glmerControl]{lme4::glmerControl}}.

 N.B. Ignored if fitting \code{\link[stats]{lm}} or \code{\link[stats]{glm}} models.}

\item{REML}{Restricted Maximum Likelihood. (Logical)}

\item{cutoff}{Threshold for predicted classes. (Numeric)

 N.B. \strong{Binomial models only}}

\item{positive}{Level from dependent variable to predict.
 Either as character or level index (\code{1} or \code{2} - alphabetically).

 E.g. if we have the levels \code{"cat"} and \code{"dog"} and we want \code{"dog"} to be the positive class,
 we can either provide \code{"dog"} or \code{2}, as alphabetically, \code{"dog"} comes after \code{"cat"}.

 Used when calculating confusion matrix metrics and creating ROC curves.

 N.B. Only affects evaluation metrics, not the model training or returned predictions.

 N.B. \strong{Binomial models only}.}

\item{metrics}{List for enabling/disabling metrics.

  E.g. \code{list("RMSE" = FALSE)} would remove RMSE from the results,
  and \code{list("Accuracy" = TRUE)} would add the regular accuracy metric
  to the classification results.
  Default values (TRUE/FALSE) will be used for the remaining metrics available.

  Also accepts the string \code{"all"}.

  N.B. Currently, disabled metrics are still computed.}

\item{err_nc}{Raise error if model does not converge. (Logical)}

\item{rm_nc}{Remove non-converged models from output. (Logical)}

\item{parallel}{Whether to validate the list of models in parallel. (Logical)

 Remember to register a parallel backend first.
 E.g. with \code{doParallel::registerDoParallel}.}

\item{model_verbose}{Message name of used model function on each iteration. (Logical)}
}
\value{
List containing tbl (tibble) with results and the trained model object.
 The tibble contains:

 \subsection{Shared across families}{

 A nested tibble with \strong{coefficients} of the models from all iterations.

 Count of \strong{convergence warnings}. Consider discarding models that did not converge on all
 iterations. Note: you might still see results, but these should be taken with a grain of salt!

 Count of \strong{other warnings}. These are warnings without keywords such as "convergence".

 Count of \strong{Singular Fit messages}. See
 \code{?\link[lme4:isSingular]{lme4::isSingular}} for more information.

 Nested tibble with the \strong{warnings and messages} caught for each model.

 Specified \strong{family}.

 Specified \strong{link} function.

 Name of \strong{dependent} variable.

 Names of \strong{fixed} effects.

 Names of \strong{random} effects, if any.
 }

 ----------------------------------------------------------------

 \subsection{Gaussian Results}{

 ----------------------------------------------------------------

 \strong{RMSE}, \strong{MAE}, \strong{r2m}, \strong{r2c}, \strong{AIC}, \strong{AICc},
 and \strong{BIC}.

 A nested tibble with the \strong{predictions} and targets.
 }

 ----------------------------------------------------------------

 \subsection{Binomial Results}{

 ----------------------------------------------------------------

 Based on predictions of the test set,
 a confusion matrix and ROC curve are used to get the following:

 ROC:

 \strong{AUC}, \strong{Lower CI}, and \strong{Upper CI}

 Confusion Matrix:

 \strong{Balanced Accuracy}, \strong{F1},
 \strong{Sensitivity}, \strong{Specificity},
 \strong{Positive Prediction Value},
 \strong{Negative Prediction Value},
 \strong{Kappa},
 \strong{Detection Rate},
 \strong{Detection Prevalence},
 \strong{Prevalence}, and
 \strong{MCC} (Matthews correlation coefficient).

 Other available metrics (disabled by default, see \code{metrics}):
 \strong{Accuracy}.

 Also includes:

 A tibble with \strong{predictions}, predicted classes (depends on \code{cutoff}),
 and the targets.

 A tibble with the sensativities and specificities from the \strong{ROC} curve.

 }
}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("stable")}

 Train gaussian or binomial models on a full training set and validate it by
 predicting the test/validation set.
 Returns results in a tibble for easy reporting, along with the trained models.
}
\details{
Packages used:

 \subsection{Models}{

 Gaussian: \link[stats:lm]{stats::lm}, \code{\link[lme4:lmer]{lme4::lmer}}

 Binomial: \code{\link[stats:glm]{stats::glm}}, \code{\link[lme4:glmer]{lme4::glmer}}
 }

 \subsection{Results}{
 \strong{Gaussian}:

 r2m : \code{\link[MuMIn:r.squaredGLMM]{MuMIn::r.squaredGLMM}}

 r2c : \code{\link[MuMIn:r.squaredGLMM]{MuMIn::r.squaredGLMM}}

 AIC : \code{\link[stats:AIC]{stats::AIC}}

 AICc : \code{\link[MuMIn:AICc]{MuMIn::AICc}}

 BIC : \code{\link[stats:BIC]{stats::BIC}}

 \strong{Binomial}:

 Confusion matrix: \code{\link[caret:confusionMatrix]{caret::confusionMatrix}}

 ROC: \code{\link[pROC:roc]{pROC::roc}}

 MCC: \code{\link[mltools:mcc]{mltools::mcc}}
 }
}
\examples{
# Attach packages
library(cvms)
library(groupdata2) # partition()
library(dplyr) # \%>\% arrange()

# Data is part of cvms
data <- participant.scores

# Set seed for reproducibility
set.seed(7)

# Partition data
# Keep as single data frame
# We could also have fed validate() separate train and test sets.
data_partitioned <- partition(data,
                              p = 0.7,
                              cat_col = 'diagnosis',
                              id_col = 'participant',
                              list_out=FALSE) \%>\%
    arrange(.partitions)

# Validate a model

# Gaussian
validate(data_partitioned,
         models = "score~diagnosis",
         partitions_col = '.partitions',
         family='gaussian',
         REML = FALSE)

# Binomial
validate(data_partitioned,
         models = "diagnosis~score",
         partitions_col = '.partitions',
         family='binomial')

# Use non-default link functions

validate(data_partitioned,
         models = "score~diagnosis",
         partitions_col = '.partitions',
         family = 'gaussian',
         link = 'log',
         REML = FALSE)

## Feed separate train and test sets

# Partition data to list of data frames
# The first data frame will be train (70\% of the data)
# The second will be test (30\% of the data)
data_partitioned <- partition(data, p = 0.7,
                              cat_col = 'diagnosis',
                              id_col = 'participant',
                              list_out=TRUE)
train_data <- data_partitioned[[1]]
test_data <- data_partitioned[[2]]

# Validate a model

# Gaussian
validate(train_data,
         test_data = test_data,
         models = "score~diagnosis",
         family='gaussian',
         REML = FALSE)

}
\seealso{
Other validation functions: \code{\link{cross_validate_fn}},
  \code{\link{cross_validate}}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{validation functions}
