context("Subvariables")

with(fake.HTTP, {
    test.ds <- loadDataset("test ds")
    mr <- test.ds$mymrset
    
    test_that("setup", {
        expect_true(is.Multiple(mr))
    })
    
    test_that("subvariables are what we think", {
        expect_true(inherits(subvariables(mr), "Subvariables"))
        expect_identical(names(subvariables(mr)), c("First", "Second", "Last"))
    })
    
    test_that("subvariable name setter error checking", {
        expect_error(names(subvariables(mr)) <- 1:3)
        expect_error(names(subvariables(mr)) <- c("First", "Second"))
        expect_error(names(subvariables(mr)) <- c("First", "First", "First"))
    })
    
    test_that("[.Subvariables", {
        expect_true(inherits(subvariables(mr)[1:2], "Subvariables"))
        expect_true(inherits(subvariables(mr)[c("First", "Last")],
            "Subvariables"))
        expect_error(subvariables(mr)[c("First", "Other")],
            "Undefined subvariables selected")
    })
    
    test_that("subvariable setter validation", {
        expect_error(subvariables(mr) <- Subvariables(), 
            "Can only reorder, not change, subvariables")
        expect_error(subvariables(mr) <- subvariables(mr)[1:2], 
            "Can only reorder, not change, subvariables")
    })
    
    test_that("Assinging in with no changes does not make PATCH request", {
        expect_that(subvariables(mr) <- subvariables(mr),
            does_not_throw_error())
    })
    
    test_that("can extract a subvariable as a Variable", {
        expect_true(inherits(subvariables(mr)[[1]], "CrunchVariable"))
        expect_true(is.Categorical(subvariables(mr)[[1]]))
        expect_true(inherits(subvariables(mr)[["Second"]], "CrunchVariable"))
        expect_true(is.Categorical(subvariables(mr)[["Second"]]))
        expect_true(inherits(subvariables(mr)$Second, "CrunchVariable"))
        expect_true(is.Categorical(subvariables(mr)$Second))
        expect_true(is.null(subvariables(mr)$Other))
    })
    
    test_that("can extract directly from array variable", {
        expect_true(inherits(mr[[1]], "CrunchVariable"))
        expect_true(is.Categorical(mr[[1]]))
        expect_true(inherits(mr[["Second"]], "CrunchVariable"))
        expect_true(is.Categorical(mr[["Second"]]))
        expect_true(inherits(mr$Second, "CrunchVariable"))
        expect_true(is.Categorical(mr$Second))
        expect_true(is.null(mr$Other))
        
        expect_true(inherits(mr[1:2], "Subvariables"))
        expect_true(inherits(mr[c("First", "Last")],
            "Subvariables"))
        expect_error(mr[c("First", "Other")],
            "Undefined subvariables selected")
    })
    
    test_that("show method for Subvariables", {
        mr <- refresh(mr)
        expect_identical(showSubvariables(subvariables(mr)), c(
            "Subvariables:",
            "  $`First`",
            "  $`Second`",
            "  $`Last`"))
    })
})

if (run.integration.tests) {
    with(test.authentication, {
        with(test.dataset(mrdf), {
            ds <- mrdf.setup(ds, selections="1.0")
            var <- ds$MR
            test_that("setup test case 2", {
                expect_true(is.Multiple(var))
                expect_identical(names(subvariables(var)),
                    c("mr_1", "mr_2", "mr_3"))
            })
            
            test_that("can rename subvariables", {
                try(names(subvariables(var))[2] <- "M.R. Two")
                expect_identical(names(subvariables(var)),
                    c("mr_1", "M.R. Two", "mr_3"))
            })
            
            test_that("can rename one subvariable", {
                try(name(subvariables(var)[[2]]) <- "Due")
                expect_identical(names(subvariables(var)),
                    c("mr_1", "Due", "mr_3"))
                sv <- subvariables(var)
                try(name(sv[[2]]) <- "M.R. Two")
                expect_identical(names(sv),
                    c("mr_1", "M.R. Two", "mr_3"))
                expect_error(name(sv[[4]]) <- "Four",
                    "subscript out of bounds")
                expect_error(sv[[2]] <- ds$v4,
                    "Cannot add or remove subvariables")
                expect_error(sv[[2]] <- NULL,
                    "Cannot add or remove subvariables")
                expect_error(sv[[2]] <- "not a variable", 
                    "Can only assign Variables into an object of class Subvariables")
            })
            test_that("can rename some subvariables", {
                try(names(subvariables(var)[2:3]) <- c("Dois", "Tres"))
                expect_identical(names(subvariables(var)),
                    c("mr_1", "Dois", "Tres"))
                sv <- subvariables(var)
                try(names(sv[2:3]) <- c("M.R. Two", "mr_3"))
                expect_identical(names(sv),
                    c("mr_1", "M.R. Two", "mr_3"))
                expect_error(names(sv[3:4]) <- c("3", "4"),
                    "Subscript out of bounds: 4")
                expect_error(sv[2:3] <- c("not a variable", "nor this"), 
                    "Can only assign Variables into an object of class Subvariables")
            })
            test_that("subvariables aliases", {
                expect_identical(aliases(subvariables(var)),
                    c("mr_1", "mr_2", "mr_3"))
                aliases(subvariables(var)) <- paste0("mr_", 5:7)
                expect_identical(aliases(subvariables(var)),
                    c("mr_5", "mr_6", "mr_7"))
            })
            
            test_that("can reorder subvariables", {
                try(subvariables(var) <- subvariables(var)[c(3,1,2)])
                expect_identical(names(subvariables(var)),
                    c("mr_3", "mr_1", "M.R. Two"))
                try(subvariables(var)[1:2] <- subvariables(var)[c(2,1)])
                expect_identical(names(subvariables(var)),
                    c("mr_1", "mr_3", "M.R. Two"))
            })
            test_that("can't (yet) otherwise modify subvariables", {
                expect_error(subvariables(var) <- NULL,
                    "Can only assign an object of class Subvariables")
                with(test.dataset(df, "other.ds"), {
                    fake <- Subvariables(allVariables(other.ds)[1:3])
                    expect_error(subvariables(var) <- fake,
                        "Can only reorder, not change, subvariables")
                    expect_error(subvariables(var)[1:2] <- fake[1:2],
                        "Cannot add or remove subvariables")
                })
            })
        })
        with(test.dataset(mrdf["mr_1"]), {
            ds <- mrdf.setup(ds)
            
            test_that("Setup for tests with array with one subvar", {
                expect_identical(length(subvariables(ds$CA)), 1L)
                expect_identical(names(subvariables(ds$CA)), "mr_1")
                expect_identical(names(categories(ds$CA)), 
                    c("0.0", "1.0", "No Data"))
            })
            
            test_that("Can edit category names", {
                names(categories(ds$CA))[1:2] <- c("False", "True")
                expect_identical(names(categories(ds$CA)), 
                    c("False", "True", "No Data"))
            })
            
            test_that("Can edit name of single-subvar", {
                names(subvariables(ds$CA)) <- "MR_1"
                expect_identical(names(subvariables(ds$CA)), "MR_1")
            })
        })
    })
}